//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright (C) 2013, 2014, 2019. Charles W. Rapp.
// All Rights Reserved.
//

package net.sf.eBus.messages.type;

import java.io.File;
import java.nio.BufferOverflowException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.util.Formatter;
import net.sf.eBus.messages.type.MessageType.MessageField;

/**
 * This class is able to serialize and de-serialize a
 * {@link java.io.File file} instance so that it may be sent
 * via an {@link net.sf.eBus.messages.EMessage eBus message}.
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public final class FileType
    extends DataType
{
//---------------------------------------------------------------
// Member data.
//

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new file data type serializer.
     */
    /* package */ FileType()
    {
        super (File.class, true, VARIABLE_SIZE, null);
    } // end of FileType()

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // DataType Abstract Method Overrides.
    //

    /**
     * Serializes the file instance to the given buffer. Writes
     * nothing if {@code o} is {@code null} or is not a
     * {@code java.io.File} instance. Serializes the file path
     * as a {@link StringType} to the buffer.
     * @param o serialize this file instance.
     * @param buffer write to this buffer.
     * @throws BufferOverflowException
     * if either {@code o} file path is longer than
     * {@link #MAX_FILE_NAME_LENGTH} or {@code buffer} does not
     * contain sufficient space to hold the path.
     */
    @Override
    public void serialize(final Object o,
                          final ByteBuffer buffer)
        throws BufferOverflowException
    {
        if (o instanceof File)
        {
            STRING_TYPE.serialize(((File) o).getPath(),
                                  MAX_FILE_NAME_LENGTH,
                                  buffer);
        }

        return;
    } // end of serialize(Object, ByteBuffer)

    /**
     * Returns the file instance de-serialized from the given
     * buffer. May return {@code null}.
     * @param buffer read in the file from this buffer.
     * @return the file object.
     * @throws BufferUnderflowException
     * if {@code buffer} contains an incomplete file instance.
     */
    @Override
    public Object deserialize(final ByteBuffer buffer)
        throws BufferUnderflowException
    {
        final String s =
            (String) STRING_TYPE.deserialize(buffer);
        Object retval = null;

        if (!s.isEmpty())
        {
            retval = new File(s);
        }

        return (retval);
    } // end of deserialize(ByteBuffer)

    /**
     * Adds the Java code used to serialize the file name to
     * a {@link ByteBuffer}.
     * @param field message field.
     * @param fieldName the message field name.
     * @param indent indent the code by this amount.
     * @param output write the code to this formatter.
     */
    @Override
    protected void createSerializer(final MessageField field,
                                    final String fieldName,
                                    final String indent,
                                    final Formatter output)
    {
        output.format(
            "%sfinal String s = (%s).getPath();%n",
            indent,
            fieldName);
        STRING_TYPE.createSerializer(
            field, "s", indent, "MAX_FILE_NAME_LENGTH", output);

        return;
    } // end of createSerializer(...)

    /**
     * Generates the code to de-serialize the {@code File}
     * instance from a {@link ByteBuffer}.
     * @param field message field.
     * @param fieldName store the de-serialized field in this
     * name.
     * @param indent indent the code by this amount.
     * @param output append the code to this formatter.
     * @param useBuilder if {@code true} then {@code fieldName}
     * is a builder method name; otherwise a local variable.
     */
    @Override
    protected void createDeserializer(final MessageField field,
                                      final String fieldName,
                                      final String indent,
                                      final Formatter output,
                                      final boolean useBuilder)
    {
        final String indent1 = indent + INDENT;
        final String format =
            (useBuilder ?
             "%sbuilder.%s(new java.io.File(s));%n" :
             "%s%s = new java.io.File(s);%n");

        output.format("%sfinal String s;%n", indent);
        STRING_TYPE.createDeserializer(
            field, "s", indent, output, false);
        output.format("%sif (s.isEmpty()) {%n", indent)
              .format("%sbuilder.%s(null);%n", indent1, fieldName)
              .format("%s} else {%n", indent)
              .format(format, indent1, fieldName)
              .format("%s}%n", indent);

        return;
    } // end of createDeserializer(...)

    //
    // end of DataType Abstract Method Overrides.
    //-----------------------------------------------------------
} // end of class FileType
