//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2017, 2019. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.messages.type;

import java.nio.BufferOverflowException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.util.Formatter;
import net.sf.eBus.messages.EField;
import net.sf.eBus.messages.EFieldList;
import net.sf.eBus.messages.type.MessageType.MessageField;

/**
 * Provides binary serialization/de-serialization for
 * {@link EFieldList}. The serialized field list format is:
 * <br>
 * <table border="1" cellpadding="3" cellspacing="3">
 *   <caption>EMessageList Serialization Layout</caption>
 *   <thead>
 *     <tr>
 *       <th>Position</th>
 *       <th>Size (bytes)</th>
 *       <th>Description</th>
 *     </tr>
 *   </thead>
 *   <tbody>
 *     <tr>
 *       <td>0</td>
 *       <td>2</td>
 *       <td>Message list size</td>
 *     </tr>
 *     <tr>
 *       <td>2</td>
 *       <td>m</td>
 *       <td>field class name</td>
 *     </tr>
 *     <tr>
 *       <td>2+m</td>
 *       <td>n</td>
 *       <td>Each {@link EField} in list</td>
 *     </tr>
 *   </tbody>
 * </table>
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public final class FieldListType
    extends DataType
{
//---------------------------------------------------------------
// Member data.
//

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new field list type instance.
     */
    /* package*/ FieldListType()
    {
        super (EFieldList.class, true, VARIABLE_SIZE, null);
    } // end of FieldListType()

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // DataType Abstract Method Overrides.
    //

    /**
     * Serializes the {@code EFieldList} instance to
     * {@code buffer}. The serialize format is show in
     * {@code EFieldListType} class documentation.
     * @param o a {@code EFieldList} instance.
     * @param buffer write output to this buffer.
     * @throws BufferOverflowException
     * if {@code buffer} does not have sufficient space to store
     * the serialized message list.
     */
    @Override
    @SuppressWarnings ("unchecked")
    public void serialize(final Object o,
                          final ByteBuffer buffer)
        throws BufferOverflowException
    {
        if (o != null)
        {
            final EFieldList fl = (EFieldList) o;

            // Is this list empty?
            if (fl.isEmpty())
            {
                // Yes. Then write out a zero list size only.
                buffer.putShort((short) 0);
            }
            else
            {
                Class<? extends EField> fc = fl.elementClass();
                final DataType dt = DataType.findType(fc);

                // Firstly, output the list size.
                buffer.putShort((short) fl.size());

                // Secondly, output the EField class.
                CLASS_TYPE.serialize(fc, buffer);

                // Thirdly, output each field in turn using the
                // field's data type.
                fl.forEach(
                    field -> { dt.serialize(field, buffer); });
            }

            // Once serialized, a field list may not be modified.
            fl.setReadOnly();
        }

        return;
    } // end of serialize(Object, ByteBuffer)

    /**
     * Returns the field list instance de-serialized from the
     * given buffer.
     * @param buffer read in the field list instance from this
     * buffer.
     * @return de-serialized field list.
     * @throws BufferUnderflowException
     * if {@code buffer} contains an incomplete field list.
     */
    @Override
    @SuppressWarnings ("unchecked")
    public Object deserialize(final ByteBuffer buffer)
        throws BufferUnderflowException
    {
        final int size = buffer.getShort();
        EFieldList retval = new EFieldList();

        if (size < 0)
        {
            final BufferUnderflowException bufex =
                new BufferUnderflowException();

            bufex.initCause(
                new IllegalArgumentException(
                    "invalid field list size " + size));

            throw (bufex);
        }
        // Are there fields to de-serialize?
        else if (size > 0)
        {
            final Class<? extends EField> fc =
                (Class<? extends EField>)
                    CLASS_TYPE.deserialize(buffer);
            final DataType dt = DataType.findType(fc);
            int index;

            for (index = 0; index < size; ++index)
            {
                retval.add((EField) dt.deserialize(buffer));
            }
        }
        // Else return an empty list.

        // Once serialized, a field list may not be modified.
        retval.setReadOnly();

        return (retval);
    } // end of deserialize(ByteBuffer)

    /**
     * Adds the Java code used to serialize a field list to a
     * {@link ByteBuffer}.
     * @param field message field.
     * @param fieldName field list field name.
     * @param indent indent the code by this white space.
     * @param output write the code to this formatter.
     */
    @Override
    protected void createSerializer(final MessageField field,
                                    final String fieldName,
                                    final String indent,
                                    final Formatter output)
    {
        output.format(
            "%s(net.sf.eBus.messages.type.DataType.findType(net.sf.eBus.messages.EFieldList.class)).serialize(%s, buffer);%n",
            indent,
            fieldName);

        return;
    } // end of createSeriliazer(...)

    /**
     * Adds the Java code used to de-serialize a field list
     * from a {@link ByteBuffer}.
     * @param field message field.
     * @param fieldName field list field name.
     * @param indent indent code by this white space.
     * @param output write the code to this formatter.
     * @param useBuilder if {@code true} then {@code fieldName}
     * is a builder method name; otherwise a local variable.
     */
    @Override
    protected void createDeserializer(final MessageField field,
                                      final String fieldName,
                                      final String indent,
                                      final Formatter output,
                                      final boolean useBuilder)
    {
        final String format =
            (useBuilder ?
             "%sbuilder.%s((%s) ((net.sf.eBus.messages.type.FieldListType) net.sf.eBus.messages.type.DataType.findType(net.sf.eBus.messages.EFieldList.class)).deserialize(buffer));%n" :
             "%s%s = (%s) ((net.sf.eBus.messages.type.FieldListType) net.sf.eBus.messages.type.DataType.findType(net.sf.eBus.messages.EFieldList.class)).deserialize(buffer);%n");

        output.format(
            format, indent, fieldName, mClass.getName());

        return;
    } // end of createDeserializer(...)

    //
    // end of DataType Abstract Method Overrides.
    //-----------------------------------------------------------
} // end of class FieldListType
