//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright (C) 2004 - 2008, 2011, 2013, 2014, 2019. Charles W. Rapp.
// All Rights Reserved.
//

package net.sf.eBus.messages.type;

import java.nio.BufferOverflowException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.util.Formatter;
import net.sf.eBus.messages.type.MessageType.MessageField;

/**
 * Provides binary serialization/de-serialization of
 * {@code java.lang.Enum} instances.
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public final class EnumType
    extends DataType
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * An {@code enum} requires {@value} bytes for encoding.
     */
    public static final int BYTES = 2;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates an Enum data type instance.
     * @param clazz the {@code Enum} subclass.
     */
    /* package */ EnumType(final Class<?> clazz)
    {
        super (clazz, true, BYTES, null);
    } // end of EnumType(Class)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // DataType Abstract Method Overrides.
    //

    /**
     * Returns {@code true} if {@code o} is a non-{@code null}
     * enum instance and {@code false} otherwise.
     * @param o Check this object's type.
     * @return {@code true} if {@code o} is a non-{@code null}
     * enum instance and {@code false} otherwise.
     */
    @Override
    public boolean isInstance(final Object o)
    {
        return (o == null || mClass.isInstance(o));
    } // end of isInstance(Object)

    /**
     * Serializes this {@code Enum} ordinal to the given
     * {@code buffer}.
     * @param o A {@code Enum} object.
     * @param buffer serialize to this buffer.
     * @exception BufferOverflowException
     * if {@code buffer} overflows.
     * @see #deserialize(ByteBuffer)
     */
    @Override
    public void serialize(final Object o, final ByteBuffer buffer)
        throws BufferOverflowException
    {
        if (o instanceof Enum)
        {
            // Serialize the enum's ordinal value.
            buffer.putShort((short) ((Enum) o).ordinal());
        }

        return;
    } // end of serialize(Object, ByteBuffer)

    /**
     * Returns the deserialized {@code Enum} in
     * {@code buffer}.
     * @param buffer Deserialize the {@code Enum} object from
     * this {@code ByteBuffer}.
     * @return the deserialized {@code Enum}.
     * @exception BufferUnderflowException
     * if {@code buffer} contains an incomplete serialized enum.
     * @see #serialize(Object, ByteBuffer)
     */
    @Override
    @SuppressWarnings("unchecked")
    public Object deserialize(final ByteBuffer buffer)
        throws BufferUnderflowException
    {
        final int ordinal = (int) buffer.getShort();

        return ((mClass.getEnumConstants())[ordinal]);
    } // end of deserialize(ByteBuffer)

    /**
     * Adds the Java code used to serialize the enum ordinal
     * value to a {@link ByteBuffer}.
     * @param field message field.
     * @param fieldName the message field name.
     * @param indent indent the code by this amount.
     * @param output write the code to this formatter.
     */
    @Override
    protected void createSerializer(final MessageField field,
                                    final String fieldName,
                                    final String indent,
                                    final Formatter output)
    {
        // Serialize ordinal value as short.
        output.format(
            "%sbuffer.putShort((short) (%s).ordinal());%n",
            indent,
            fieldName);

        return;
    } // end of createSerializer(...)

    /**
     * Generates the code to de-serialize the enum instance
     * from a {@link ByteBuffer} based on the enum ordinal value.
     * @param field message field.
     * @param fieldName store the de-serialized field in this
     * name.
     * @param indent indent the code by this amount.
     * @param output append the code to this formatter.
     * @param useBuilder if {@code true} then {@code fieldName}
     * is a builder method name; otherwise a local variable.
     */
    @Override
    protected void createDeserializer(final MessageField field,
                                      final String fieldName,
                                      final String indent,
                                      final Formatter output,
                                      final boolean useBuilder)
    {
        final String format =
            (useBuilder ?
             "%sbuilder.%s((%s.values())[(int) buffer.getShort()]);%n" :
             "%s%s = (%s.values())[(int) buffer.getShort()];%n");

        output.format(
            format, indent, fieldName, mClass.getCanonicalName());

        return;
    } // end of createDeserializer(...)

    //
    // end of DataType Abstract Method Overrides.
    //-----------------------------------------------------------
} // end of class EnumType
