//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright (C) 2004 - 2008, 2011, 2013, 2014, 2019. Charles W. Rapp.
// All Rights Reserved.
//

package net.sf.eBus.messages.type;

import java.nio.BufferOverflowException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.util.Date;
import java.util.Formatter;
import net.sf.eBus.messages.type.MessageType.MessageField;

/**
 * Provides binary serialization/de-serialization of
 * {@code java.util.Date} values.
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public final class DateType
    extends DataType
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * Date type serialized size is {@link Long#BYTES}.
     */
    public static final int BYTES = Long.BYTES;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates the date type object.
     */
    /* package */ DateType()
    {
        super (Date.class, true, BYTES, null);
    } // end of DateType()

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // DataType Abstract Method Overrides.
    //

    /**
     * Serializes this {@code Date} to the given {@code buffer}.
     * @param o a {@code Date} object.
     * @param buffer write to this buffer.
     * @exception BufferOverflowException
     * if {@code buffer} overflows.
     * @see #deserialize(ByteBuffer)
     */
    @Override
    public void serialize(final Object o,
                          final ByteBuffer buffer)
        throws BufferOverflowException
    {
        if (o instanceof Date)
        {
            buffer.putLong(((Date) o).getTime());
        }

        return;
    } // end of serialize(Object, ByteBuffer)

    /**
     * Returns the deserialized {@code Date} in
     * {@code buffer}.
     * @param buffer Deserialize the date object from this
     * {@code ByteBuffer}.
     * @return the deserialized {@code Date}.
     * @exception BufferUnderflowException
     * if {@code buffer} contains an incomplete serialized date.
     * @see #serialize(Object, ByteBuffer)
     */
    @Override
    public Object deserialize(final ByteBuffer buffer)
        throws BufferUnderflowException
    {
        return (new Date(buffer.getLong()));
    } // end of deserialize(ByteBuffer)

    /**
     * Adds the Java code used to serialize the Date instance to
     * a {@link ByteBuffer}.
     * @param field message field.
     * @param fieldName the message field name.
     * @param indent indent the code by this amount.
     * @param output write the code to this formatter.
     */
    @Override
    protected void createSerializer(final MessageField field,
                                    final String fieldName,
                                    final String indent,
                                    final Formatter output)
    {
        output.format(
            "%sbuffer.putLong((%s).getTime());%n",
            indent,
            fieldName);

        return;
    } // end of createSerializer(...)

    /**
     * Generates the code to de-serialize the {@code Date}
     * instance from a {@link ByteBuffer}.
     * @param field message field.
     * @param fieldName store the de-serialized field in this
     * name.
     * @param indent indent the code by this amount.
     * @param output append the code to this formatter.
     * @param useBuilder if {@code true} then {@code fieldName}
     * is a builder method name; otherwise a local variable.
     */
    @Override
    protected void createDeserializer(final MessageField field,
                                      final String fieldName,
                                      final String indent,
                                      final Formatter output,
                                      final boolean useBuilder)
    {
        final String format =
            (useBuilder ?
             "%sbuilder.%s(new java.util.Date(buffer.getLong()));%n" :
             "%s%s = new java.util.Date(buffer.getLong());%n");

        output.format(format, indent, fieldName);

        return;
    } // end of createDeserializer(...)

    //
    // end of DataType Abstract Method Overrides.
    //-----------------------------------------------------------
} // end of class DateType
