//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright (C) 2004 - 2008, 2011, 2013 - 2017, 2019. Charles W. Rapp.
// All Rights Reserved.
//

package net.sf.eBus.messages.type;

import java.io.File;
import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodHandles;
import java.lang.invoke.MethodType;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.URI;
import java.nio.BufferOverflowException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.MonthDay;
import java.time.OffsetDateTime;
import java.time.OffsetTime;
import java.time.Period;
import java.time.Year;
import java.time.YearMonth;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Formatter;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import javassist.CannotCompileException;
import javassist.ClassPool;
import javassist.CtClass;
import javassist.CtNewConstructor;
import javassist.CtNewMethod;
import javassist.NotFoundException;
import net.sf.eBus.messages.EFieldList;
import net.sf.eBus.messages.ELocalOnly;
import net.sf.eBus.messages.EMessage;
import net.sf.eBus.messages.EMessageKey;
import net.sf.eBus.messages.EMessageList;
import net.sf.eBus.messages.EMessageObject;
import net.sf.eBus.messages.EReplyInfo;
import net.sf.eBus.messages.EReplyMessage;
import net.sf.eBus.messages.ERequestMessage;
import net.sf.eBus.messages.InvalidMessageException;
import net.sf.eBus.messages.ValidationException;
import net.sf.eBus.util.MultiKey2;

/**
 * All message fields have an associated {@code DataType}
 * subclass which is responsible for serializing and
 * deserializing field values. Both binary and text serialization
 * is supported. This class maintains a Java class-to-DataType
 * instance map. Thus for each
 * {@link net.sf.eBus.messages.EField} class, there is only one
 * data type instance used for serialization. The data type
 * instances are created only as needed and only once.
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public abstract class DataType
    implements Comparable<DataType>
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * An array or collection data type may contain at most
     * 1,000 elements.
     */
    public static final int MAX_ARRAY_SIZE = 1000;

    /**
     * Strings may be at most 1,024 characters.
     */
    public static final int MAX_STRING_LENGTH = 1024;

    /**
     * The maximum file name length is 1,024 characters.
     */
    public static final int MAX_FILE_NAME_LENGTH = 1024;

    /**
     * A message object may have at most 31 fields. This is
     * because there are 31 usable bits in a 4-byte, signed
     * integer.
     */
    public static final int MAX_FIELDS = 31;

    /**
     * eBus uses {@link StandardCharsets#UTF_8} to serialize and
     * de-serialize strings.
     */
    public static final Charset CHARSET = StandardCharsets.UTF_8;

    /**
     * A Type name ending in "[]" is an array.
     */
    public static final String ARRAY_SUFFIX = "[]";

    /**
     * When a data type's size is variable then set the size
     * to {@link Integer#MAX_VALUE}.
     */
    public static final int VARIABLE_SIZE = Integer.MAX_VALUE;

    /**
     * The field mask is a 4-byte, signed integer.
     */
    protected static final int FIELD_MASK_SIZE = 4;

    /**
     * The default return value for text serialization is an
     * empty string ("").
     */
    protected static final String EMPTY_STRING = "";

    /**
     * Use two blanks as generated retval indent.
     */
    protected static final String INDENT = "  ";

    /**
     * A four blank indent used for fields.
     */
    protected static final String INDENT1 = "    ";

    /**
     * Used by subclass for {@code String}
     * serialization/de-serialization.
     */
    protected static final StringType STRING_TYPE;

    /**
     * The Internet address type is used to serialize and
     * de-serialize the address portion of the socket address.
     */
    protected static final InetAddressType ADDRESS_TYPE;

    /**
     * Used to serialize/de-serialize a Java class.
     */
    protected static final ClassType CLASS_TYPE;

    /**
     * Used to serialize/de-serialize an eBus message key.
     */
    protected static final MessageKeyType KEY_TYPE;

    //-----------------------------------------------------------
    // Statics.
    //

    /**
     * Maps the Java class to the eBus data type encapsulating
     * that class.
     */
    private static final Map<Class<?>, DataType> sDataTypeMap;

    /**
     * Caches the Java class instance associated with a given
     * fully-qualified class name.
     */
    protected static final Map<String, Class<?>> sClasses;

    /**
     * Tracks which {@link EMessageObject} classes are compiled.
     */
    protected static final
        List<Class<? extends EMessageObject>> sCompiledClasses;

    static
    {
        Class<?> jType;
        Class<?> pType;

        sDataTypeMap = new ConcurrentHashMap<>();
        sClasses = new HashMap<>();
        sCompiledClasses = new ArrayList<>();

        // Initialize the data type map with the primative types
        // and their class counterparts.
        pType = boolean.class;
        sDataTypeMap.put(pType, new BooleanType(pType));
        sClasses.put(pType.getName(), pType);

        jType = Boolean.class;
        sDataTypeMap.put(jType, new BooleanType(jType));
        sClasses.put(jType.getName(), jType);

        pType = byte.class;
        sDataTypeMap.put(pType, new ByteType(pType));
        sClasses.put(pType.getName(), pType);

        jType = Byte.class;
        sDataTypeMap.put(jType, new ByteType(jType));
        sClasses.put(jType.getName(), jType);

        pType = char.class;
        sDataTypeMap.put(pType, new CharType(pType));
        sClasses.put(pType.getName(), pType);

        jType = Character.class;
        sDataTypeMap.put(jType, new CharType(jType));
        sClasses.put(jType.getName(), jType);

        pType = double.class;
        sDataTypeMap.put(pType, new DoubleType(pType));
        sClasses.put(pType.getName(), pType);

        jType = Double.class;
        sDataTypeMap.put(jType, new DoubleType(jType));
        sClasses.put(jType.getName(), jType);

        pType = float.class;
        sDataTypeMap.put(pType, new FloatType(pType));
        sClasses.put(pType.getName(), pType);

        jType = Float.class;
        sDataTypeMap.put(jType, new FloatType(jType));
        sClasses.put(jType.getName(), jType);

        pType = int.class;
        sDataTypeMap.put(pType, new IntType(pType));
        sClasses.put(pType.getName(), pType);

        jType = Integer.class;
        sDataTypeMap.put(jType, new IntType(jType));
        sClasses.put(jType.getName(), jType);

        pType = long.class;
        sDataTypeMap.put(pType, new LongType(pType));
        sClasses.put(pType.getName(), pType);

        jType = Long.class;
        sDataTypeMap.put(jType, new LongType(jType));
        sClasses.put(jType.getName(), jType);

        pType = short.class;
        sDataTypeMap.put(pType, new ShortType(pType));
        sClasses.put(pType.getName(), pType);

        jType = Short.class;
        sDataTypeMap.put(jType, new ShortType(jType));
        sClasses.put(jType.getName(), jType);

        //
        // The remaining have no primative counterpart.
        //

        jType = java.util.Date.class;
        sDataTypeMap.put(jType, new DateType());
        sClasses.put(jType.getName(), jType);

        jType = String.class;
        STRING_TYPE = new StringType();
        sDataTypeMap.put(jType, STRING_TYPE);
        sClasses.put(jType.getName(), jType);

        jType = URI.class;
        sDataTypeMap.put(jType, new UriType());
        sClasses.put(jType.getName(), jType);

        jType = File.class;
        sDataTypeMap.put(jType, new FileType());
        sClasses.put(jType.getName(), jType);

        jType = BigInteger.class;
        sDataTypeMap.put(jType, new BigIntegerType());
        sClasses.put(jType.getName(), jType);

        jType = BigDecimal.class;
        sDataTypeMap.put(jType, new BigDecimalType());
        sClasses.put(jType.getName(), jType);

        jType = Class.class;
        CLASS_TYPE = new ClassType();
        sDataTypeMap.put(jType, CLASS_TYPE);
        sClasses.put(jType.getName(), jType);

        jType = EMessageKey.class;
        KEY_TYPE = new MessageKeyType();
        sDataTypeMap.put(jType, KEY_TYPE);
        sClasses.put(jType.getName(), jType);

        jType = EMessageList.class;
        sDataTypeMap.put(jType, new MessageListType());
        sClasses.put(jType.getName(), jType);

        jType = EFieldList.class;
        sDataTypeMap.put(jType, new FieldListType());
        sClasses.put(jType.getName(), jType);

        ADDRESS_TYPE = new InetAddressType();
        jType = InetAddress.class;
        sDataTypeMap.put(jType, ADDRESS_TYPE);
        sClasses.put(jType.getName(), jType);

        jType = Inet4Address.class;
        sDataTypeMap.put(jType, ADDRESS_TYPE);
        sClasses.put(jType.getName(), jType);

        jType = Inet6Address.class;
        sDataTypeMap.put(jType, ADDRESS_TYPE);
        sClasses.put(jType.getName(), jType);

        jType = InetSocketAddress.class;
        sDataTypeMap.put(jType, new InetSocketAddressType());
        sClasses.put(jType.getName(), jType);

        // java.time classes.

        jType = Duration.class;
        sDataTypeMap.put(jType, new DurationType());
        sClasses.put(jType.getName(), jType);

        jType = Instant.class;
        sDataTypeMap.put(jType, new InstantType());
        sClasses.put(jType.getName(), jType);

        jType = LocalDate.class;
        sDataTypeMap.put(jType, new LocalDateType());
        sClasses.put(jType.getName(), jType);

        jType = LocalTime.class;
        sDataTypeMap.put(jType, new LocalTimeType());
        sClasses.put(jType.getName(), jType);

        jType = LocalDateTime.class;
        sDataTypeMap.put(jType, new LocalDateTimeType());
        sClasses.put(jType.getName(), jType);

        jType = MonthDay.class;
        sDataTypeMap.put(jType, new MonthDayType());
        sClasses.put(jType.getName(), jType);

        jType = OffsetTime.class;
        sDataTypeMap.put(jType, new OffsetTimeType());
        sClasses.put(jType.getName(), jType);

        jType = OffsetDateTime.class;
        sDataTypeMap.put(jType, new OffsetDateTimeType());
        sClasses.put(jType.getName(), jType);

        jType = Period.class;
        sDataTypeMap.put(jType, new PeriodType());
        sClasses.put(jType.getName(), jType);

        jType = Year.class;
        sDataTypeMap.put(jType, new YearType());
        sClasses.put(jType.getName(), jType);

        jType = YearMonth.class;
        sDataTypeMap.put(jType, new YearMonthType());
        sClasses.put(jType.getName(), jType);

        jType = ZonedDateTime.class;
        sDataTypeMap.put(jType, new ZonedDateTimeType());
        sClasses.put(jType.getName(), jType);

        jType = ZoneId.class;
        sDataTypeMap.put(jType, new ZoneIdType());
        sClasses.put(jType.getName(), jType);

        jType = ZoneOffset.class;
        sDataTypeMap.put(jType, new ZoneOffsetType());
        sClasses.put(jType.getName(), jType);
    } // end of static

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * The Java class represented by this data type.
     */
    protected transient Class<?> mClass;

    /**
     * This flag is {@code true} if the data type is considered a
     * built-in type.
     */
    protected final boolean mBuiltinFlag;

    /**
     * Number of bytes needed to represent this data type.
     */
    protected final int mSize;

    /**
     * If a built-in type value is not specified, then set it
     * to this default value. Will be {@code null} for all other
     * types.
     */
    protected final Object mDefaultValue;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Set this data type Java class type.
     * @param jClass the Java associated Java class.
     * @param builtinFlag {@code true} if this data type is a
     * built-in type.
     * @param size data type size in bytes.
     * @param defaultValue the default value for this type.
     */
    protected DataType(final Class<?> jClass,
                       final boolean builtinFlag,
                       final int size,
                       final Object defaultValue)
    {
        mClass = jClass;
        mBuiltinFlag = builtinFlag;
        mSize = size;
        mDefaultValue = defaultValue;
    } // end of DataType(Class, boolean, int, Object)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Abstract Methods Declarations.
    //

    /**
     * Serializes {@code o} to the given {@code buffer}.
     * @param o serialize this object.
     * @param buffer serialize {@code o} to this buffer.
     * @exception BufferOverflowException
     * if serializing {@code o} overruns {@code buffer}'s
     * available capacity.
     */
    public abstract void serialize(Object o, ByteBuffer buffer)
        throws BufferOverflowException;

    /**
     * Returns the deserialized object in {@code buffer}.
     * @param buffer Deserialize the object from this
     * {@code ByteBuffer}.
     * @return the deserialized object.
     * @exception BufferUnderflowException
     * if {@code buffer} contains fewer bytes than needed to
     * completely deserialize the object.
     * @exception ValidationException
     * if {@code buffer} contains an invalid eBus message.
     */
    public abstract Object deserialize(ByteBuffer buffer)
        throws BufferUnderflowException,
               ValidationException;

    /**
     * Adds the Java retval used to serialize the named field to
     * a {@link ByteBuffer}.
     * @param field message field.
     * @param fieldName the message field name.
     * @param indent indent the retval by this amount.
     * @param output write the retval to this formatter.
     */
    protected abstract void createSerializer(MessageType.MessageField field,
                                             String fieldName,
                                             String indent,
                                             Formatter output);

    /**
     * Adds the Java retval used to de-serialize the named field
     * from a {@link ByteBuffer}. Store the result into a
     * message/field builder if {@code useBuilder} is
     * {@code true} and into a local field otherwise.
     * @param field message field.
     * @param fieldName fully-qualified message field name.
     * @param indent indent the retval by this amount.
     * @param output write the retval to this formatter.
     * @param useBuilder if {@code true} then {@code fieldName}
     * is a builder method name; otherwise a local variable.
     */
    protected abstract void createDeserializer(MessageType.MessageField field,
                                               String fieldName,
                                               String indent,
                                               Formatter output,
                                               boolean useBuilder);

    //
    // end of Abstract Methods Declarations.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Comparable Interface Implementation.
    //

    /**
     * Returns an integer value &lt; zero, zero or &gt; 0 if
     * this data type is &lt;, equals or &gt; {@code findType}.
     * The comparison is based on the class name.
     * @param dataType compare to this data type.
     * @return an integer value &lt; zero, zero or &gt; 0 if
     * this data type is &lt;, equals or &gt; {@code findType}.
     */
    @Override
    public int compareTo(final DataType dataType)
    {
        return (
            (mClass.getName()).compareTo(
                dataType.mClass.getName()));
    } // end of compareTo(DataType)

    //
    // end of Comparable Interface Implementation.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Object Method Overrides.
    //

    /**
     * Returns {@code true} if {@code o} is not
     * {@code null}, is an instance of {@code DataType}
     * and has the same Java class. Otherwise returns
     * {@code false}.
     * @param o compare against this object.
     * @return {@code true} if {@code o} is not
     * {@code null}, is an instance of {@code DataType}
     * and has the same Java class. Otherwise returns
     * {@code false}.
     */
    @Override
    public boolean equals(final Object o)
    {
        boolean retcode = (o == this);

        if (!retcode && o instanceof DataType)
        {
            final DataType type = (DataType) o;

            retcode = mClass.equals(type.mClass);
        }

        return (retcode);
    } // end of equals(Object)

    /**
     * Returns the data type's hash retval based on the class type.
     * @return the data type's hash retval based on the class type.
     */
    @Override
    public int hashCode()
    {
        return (mClass.hashCode());
    } // end of hashCode()

    /**
     * Returns the data type's Java class name.
     * @return the data type's Java class name.
     */
    @Override
    public String toString()
    {
        return (mClass.getName());
    } // end of toString()

    //
    // end of Object Method Overrides.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get methods.
    //

    /**
     * Returns {@code true} if {@code o} is of the
     * proper type and {@code false} otherwise.
     * @param o check this object's type.
     * @return {@code true} if {@code o} is of the
     * proper type and {@code false} otherwise.
     */
    @SuppressWarnings ("unchecked")
    public boolean isInstance(final Object o)
    {
        return (o == null ||
                mClass.isAssignableFrom(o.getClass()));
    } // end of isInstance(Object)

    /**
     * Returns the associated Java class.
     * @return the associated Java class.
     */
    public final Class<?> dataClass()
    {
        return (mClass);
    } // end of dataClass()

    /**
     * Returns the Java class name.
     * @return Java class name.
     */
    public String dataClassName()
    {
        return (mClass.getCanonicalName());
    } // end of dataClassName()

    /**
     * Returns {@code true} if this data type is an array and
     * {@code false} otherwise.
     * @return {@code true} if the data type is an array.
     */
    public final boolean isArray()
    {
        return (mClass.isArray());
    } // end of isArray()

    /**
     * Returns {@code true} if this data type is considered an
     * <em>eBus</em> built-in type and {@code false} otherwise.
     * @return {@code true} if this is an eBus built-in type.
     */
    public final boolean isBuiltin()
    {
        return (mBuiltinFlag);
    } // end of isBuiltin()

    /**
     * Returns {@code true} if this data type is for a
     * <em>Java</em> primitive type and {@code false} otherwise.
     * @return {@code true} if this is a Java primitive type.
     */
    public final boolean isPrimitive()
    {
        return (mClass.isPrimitive());
    } // end of isPrimitive()

    /**
     * Returns {@code true} if this is an enum data type and
     * {@code false} otherwise.
     * @return {@code true} if this is an enum data type.
     */
    public final boolean isEnum()
    {
        return (mClass.isEnum());
    } // end of isEnum()

    /**
     * Returns {@code true} if this message type is for an
     * {@code EMessage} and {@code false} if for an
     * {@code EField}.
     * @return {@code true} if the underlying type is an
     * {@code EMessage}.
     */
    public final boolean isMessage()
    {
        return ((EMessage.class).isAssignableFrom(mClass));
    } // end of isMessage()

    /**
     * Returns {@code true} if this message type is local-only
     * and {@code false} if this message type may be transmitted
     * to remote eBus applications.
     * @return {@code true} if this is a local-only message
     * object.
     */
    public boolean isLocalOnly()
    {
        return (mClass.isAnnotationPresent(ELocalOnly.class));
    } // end of isLocalOnly()

    /**
     * Returns the data type size in bytes. Returns
     * {@link #VARIABLE_SIZE} if the type size is variable.
     * @return data type byte size.
     */
    public final int size()
    {
        return (mSize);
    } // end of size()

    /**
     * Returns the default value for this type. May return
     * {@code null}.
     * @return type default value.
     */
    public final Object defaultValue()
    {
        return (mDefaultValue);
    } // end of defaultValue()

    //
    // end of Get methods.
    //-----------------------------------------------------------

    /**
     * Returns the data type object associated with the class.
     * If no such object is cached, then attempts to create
     * a {@code DataType} for {@code jClass}. Throws either an
     * {@code IllegalArgumentException} or
     * {@code InvalidMessageException} if this cannot be done.
     * @param jClass A Java class instance.
     * @return the data type object associated with the class.
     * @throws IllegalArgumentException
     * if {@code jClass} is an unsupported eBus message type.
     * @throws InvalidMessageException
     * if {@code jClass} is an {@link EMessageObject} with an
     * invalid message layout.
     */
    @SuppressWarnings ({"unchecked"})
    public static DataType findType(final Class<?> jClass)
        throws IllegalArgumentException,
               InvalidMessageException
    {
        DataType retval = null;

        // A null class results in a null return value.
        Objects.requireNonNull(jClass, "jClass is null");

        synchronized (sDataTypeMap)
        {
            // Is this class in the map and set to null or
            // simply not in the map?
            if (sDataTypeMap.containsKey(jClass))
            {
                retval = sDataTypeMap.get(jClass);
            }
            // This must be an Object type since all Java
            // primitive types are already stored away in
            // sDataTypeMap.
            // Is this an enum subclass?
            else if (jClass.isEnum())
            {
                retval = new EnumType(jClass);
            }
            // If this is an array then create an array type but only
            // if the array elements are eBus serializable.
            // Get the element type class by stripping the "[]" from
            // the end of the array class name.
            else if (jClass.isArray())
            {
                final DataType elementType =
                    DataType.findType(jClass.getComponentType());

                if (elementType != null)
                {
                    retval = new ArrayType(jClass, elementType);
                }
            }
            // Then this object must be either a message or a
            // field. If not, then that is an error.
            else if (!EMessageObject.class.isAssignableFrom(jClass))
            {
                // no-op. Leave the return value set to null.
                // Check below will catch this.
            }
            // Is this class a local-only message or field?
            else if (jClass.isAnnotationPresent(ELocalOnly.class))
            {
                // Yes. Create a LocalMessageType which allows
                // fields to be any type supported by Java.
                retval =
                    LocalMessageType.createLocalMessageType(
                        (Class<? extends EMessageObject>) jClass);
            }
            // Is this class an abstract message or field?
            // Or is this a local-only
            else if ((jClass.getModifiers() & Modifier.ABSTRACT) ==
                         Modifier.ABSTRACT)
            {
                // Yes. Create an AbstractMesssageType and store
                // away.
                retval =
                    AbstractMessageType.createAbstractMessageType(
                        (Class<? extends EMessageObject>) jClass);
            }
            else
            {
                retval = createConcreteMessageType(jClass);
            }
        }

        // If null is returned, then throw an invalid
        // message exception explaining why there is
        // not eBus data type for jClass.
        if (retval == null)
        {
            throw (
                new IllegalArgumentException(
                    String.format(
                        "%s is not a supported in eBus",
                        jClass.getCanonicalName())));
        }

        sDataTypeMap.put(jClass, retval);

        return (retval);
    } // end of findType(Class)

    /**
     * Returns the specified message class  or field class
     * field name, data type pairs. The returned field order
     * matches defines field serialization order.
     * @param mc find the fields for this {@code EMessage} or
     * {@code EField} class.
     * @return {@code mc} field name, data type pairs.
     * @throws IllegalArgumentException
     * if {@code mc} is {@code null}.
     * @throws InvalidMessageException
     * if {@code mc} is an invalid eBus message.
     */
    public static List<MultiKey2<String, DataType>> fields(Class<? extends EMessageObject> mc)
        throws IllegalArgumentException,
               InvalidMessageException
    {
        final MessageType msgType;
        final List<MessageType.MessageField> fields;
        final int numFields;
        int index;
        MessageType.MessageField field;
        final List<MultiKey2<String, DataType>> retval;

        if (mc == null)
        {
            throw (new IllegalArgumentException("mc is null"));
        }

        msgType = (MessageType) findType(mc);
        fields = msgType.fields();
        numFields = fields.size();
        retval = new ArrayList<>(numFields);

        for (index = 0; index < numFields; ++index)
        {
            field = fields.get(index);
            retval.add(
                new MultiKey2<>(
                    field.name(), field.dataType()));
        }

        return (retval);
    } // end of fields(Class<>)

    /**
     * Creates the actual data type object based on the class.
     * Either the data type is an enum, an array, or an
     * EMessageObject instance.
     * @param jClass the Java class instance.
     * @return the eBus data type object encapsulating the Java
     * class.
     * @throws InvalidMessageException
     * if {@code jClass} is an invalid eBus message object.
     */
    @SuppressWarnings ("unchecked")
    private static DataType createConcreteMessageType(final Class<?> jClass)
        throws InvalidMessageException
    {
        final Class<? extends EMessageObject> mc =
            (Class<? extends EMessageObject>) jClass;
        DataType retval = null;

        try
        {
            // Take the concrete message type and create a
            // ConcreteMessageType subclass specifically for
            // mc.
            retval = compile(ConcreteMessageType.load(mc));
        }
        catch (IllegalAccessException |
               InstantiationException |
               CannotCompileException |
               NotFoundException jex)
        {
            throw (
                new InvalidMessageException(
                    mc,
                    "failed to create " + mc.getCanonicalName() + " type",
                    jex));
        }

        return (retval);
    } // end of createConcreteMessageType(Class)

    /**
     * Returns a {@code DataType} instance especially created to
     * serialize and de-serialize the given eBus message type.
     * This purpose built data type serializes ~ 10x faster and
     * de-serializes ~ 20x faster than the generic
     * {@link MessageType}. The reason for this performance
     * improvement is because {@code MessageType} heavily uses
     * auto-boxing and {@link java.lang.invoke.MethodHandle} to
     * construct the de-serialized message.
     * @param mt create the data type for this eBus message type.
     * @return a purpose built data type specifically created for
     * the message type.
     * @throws CannotCompileException
     * if the generated retval cannot be compiled.
     * @throws NotFoundException
     * will not be thrown because the message fields were
     * verified before this method is called.
     * @throws InstantiationException
     * will not be thrown because eBus
     * @throws IllegalAccessException
     * will not be thrown because eBus is allowed to access retval
 it generates.
     */
    private static DataType compile(final ConcreteMessageType mt)
        throws CannotCompileException,
               NotFoundException,
               InstantiationException,
               IllegalAccessException
    {
        @SuppressWarnings ("unchecked")
        final Class<? extends EMessageObject> mc =
            (Class<? extends EMessageObject>) mt.dataClass();
        final String className =
            String.format(
                "__%sType__", (mt.mClass).getSimpleName());
        final String fqName =
            String.format(
                "%s.%s",
                (mc.getPackage()).getName(),
                className);
        final String ctor =
            String.format(
                "public %s(java.util.List fields, java.util.List replies, java.lang.reflect.Method builder, java.lang.Class bc) {\n" +
                    "%ssuper (%s.class, fields, replies, builder, bc);\n" +
                    "}",
                className,
                INDENT,
                mc.getName());
        final ClassPool pool = ClassPool.getDefault();
        int numBools = 0;
        final CtClass retval = pool.makeClass(fqName);

        // Make the generate datatype class both public and
        // final.
        retval.setModifiers(javassist.Modifier.FINAL |
                            javassist.Modifier.PUBLIC);

        // The generated message type extends the eBus message
        // type.
        retval.setSuperclass(
            pool.get((ConcreteMessageType.class).getName()));

        // Generate the required constructor with super class
        // call.
        retval.addConstructor(
            CtNewConstructor.make(ctor, retval));

        // Count up the number of boolean fields. Serialization
        // and deserialization will use a boolean mask field if
        // there are any boolean fields.
        numBools = (mt.fields()).stream()
                                .filter(
                                    field ->
                                        isBoolean((field.dataType()).dataClass()))
                                .map(mItem -> 1)
                                .reduce(numBools, Integer::sum);

        // If there are boolean values in this message, then
        // generate the serializer's booleanMask() method.
        // Do this *before* definining createSerializer() because
        // that method references serializeBooleans().
        // The deserializer handles booleans within the
        // deserializeFields() method.
        if (numBools > 0)
        {
            retval.addMethod(
                CtNewMethod.make(
                    serializeBooleans(mt), retval));
        }

        // Add the required serializeFields and deserializeFields
        // methods.
        retval.addMethod(
            CtNewMethod.make(
                createSerializer(mt, numBools), retval));

        retval.addMethod(
            CtNewMethod.make(
                createDeserializer(mt, numBools), retval));

        return (instantiateMessageType(mt, retval.toClass()));
    } // end of compile(ConcreteMessageType)

    /**
     * Returns the {@code serializeFields} method for the given
     * message type.
     * @param mt the eBus message type.
     * @param numBools number of boolean fields in {@code mt}.
     * @return serializeBody method.
     */
    private static String createSerializer(final MessageType mt,
                                           final int numBools)
    {
        final List<MessageType.MessageField> fields =
            mt.fields();
        final Formatter retval = new Formatter();

        // Note: Must use fully-qualified names in the generated
        // retval 'cuz there ain't no importn'.
        retval.format("protected int serializeFields(java.lang.Object o, java.nio.ByteBuffer buffer)%n")
              .format("%sthrows java.nio.BufferOverflowException%n",
                      INDENT)
              .format("{%n")
              .format("%1s%2$s value = (%2$s) o;%n",
                      INDENT,
                      mt.dataClassName())
              .format("%sint retval = 0x%x;%n",
                      INDENT,
                      fieldMask(fields));

        // Output the boolean mask first - if there are any
        // booleans.
        if (numBools > 0)
        {
            retval.format(
                "%sbuffer.putInt(booleanMask(value));%n",
                INDENT);
        }

        retval.format("%n");

        serializeFields(retval, fields);

        // Return the field mask to the caller.
        retval.format("  return (retval);%n");
        retval.format("}");

        return (retval.toString());
    } // end of createSerializer(MessageType)

    /**
     * Generates the de-serialization code for the given
     * message type.
     * @param mt generate the de-serializer code for this
     * message type.
     * @param numBools number of boolean fields in {@code mt}.
     * @return Java de-serialization code.
     */
    private static String createDeserializer(final ConcreteMessageType mt,
                                             final int numBools)
    {
        final boolean messageFlag = mt.isMessage();
        final List<MessageType.MessageField> fields =
            mt.fields();
        final Formatter retval = new Formatter();

        retval.format("protected java.lang.Object deserializeFields(int fieldMask, java.nio.ByteBuffer buffer)%n")
              .format("%sthrows java.lang.IllegalArgumentException,%n",
                      INDENT)
              .format("%s       java.nio.BufferUnderflowException,%n",
                      INDENT)
              .format("%s       net.sf.eBus.messages.ValidationException%n",
                      INDENT)
              .format("{%n")
              .format("%sfinal %s builder = %s.builder();%n%n",
                      INDENT,
                      mt.builderClassName(),
                      mt.dataClassName());

        // If this is an EMessage de-serializer, then place
        // mSubject into the builder.
        // Note: caller has already verified that mSubject is not
        // null or empty.
        if (messageFlag)
        {
            // Store the subject into f0 and then clear the
            // subject;
            retval.format("%sbuilder.subject(mSubject);%n",
                          INDENT)
                  .format("%smSubject = null;%n%n", INDENT);
        }

        // If there are any boolean fields, the boolean mask
        // will be next.
        if (numBools > 0)
        {
            retval.format("%sfinal int boolMask = buffer.getInt();%n%n",
                          INDENT);
            deserializeBooleans(retval, fields);
        }

        // Generate code to deserialize the message fields.
        deserializeFields(retval, fields);

        // Build the message instance return value.
        retval.format("%sreturn (builder.build());%n", INDENT)
              .format("}");

        return (retval.toString());
    } // end of createDeserializer(ConcreteMessageType, int)

    /**
     * Returns the initial field mask for all <em>primitive</em>
     * fields because these field values cannot be {@code null}.
     * @param fields message fields.
     * @return initial field mask.
     */
    private static int fieldMask(final List<MessageType.MessageField> fields)
    {
        final int numFields = fields.size();
        int fieldIndex;
        DataType fieldType;
        int retval = 0;

        for (fieldIndex = 0; fieldIndex < numFields; ++fieldIndex)
        {
            fieldType = (fields.get(fieldIndex)).dataType();

            if (fieldType.isPrimitive() && !fieldType.isArray())
            {
                retval |= (1 << fieldIndex);
            }
        }

        return (retval);
    } // end of fieldMask(List<>)

    /**
     * Returns {@code true} if the given data class is either
     * {@code boolean} or {@code Boolean}.
     * @param dataClass check if this is a boolean type.
     * @return {@code true} if {@code dataClass} is a boolean
     * type.
     */
    private static boolean isBoolean(final Class<?> dataClass)
    {
        return (boolean.class.equals(dataClass) ||
                Boolean.class.equals(dataClass));
    } // end of isBoolean(Class)

    /**
     * Adds code to generate the message field serialization.
     * @param output add code to this output formatter.
     * @param fields generate serialization code for these codes.
     */
    private static void serializeFields(final Formatter output,
                                        final List<MessageType.MessageField> fields)
    {
        final int numFields = fields.size();
        int fieldIndex;
        MessageType.MessageField field;
        String fieldName;
        DataType fieldType;

        for (fieldIndex = 0; fieldIndex < numFields; ++fieldIndex)
        {
            field = fields.get(fieldIndex);
            fieldName = String.format("value.%s", field.name());
            fieldType = field.dataType();

            // Skip boolean fields as well since they are
            // already "serialized" into the boolean mask.
            if (isBoolean(fieldType.dataClass()))
            {
                // no-op.
            }
            // Primitive or non-null Object reference?
            else if (fieldType.isPrimitive() &&
                     !fieldType.isArray())
            {
                // Non-array primitive.
                fieldType.createSerializer(
                    field, fieldName, INDENT, output);
            }
            // Object reference. Check if the field is null
            // before trying to serialize it.
            else
            {
                output.format("%sif (%s != null) {%n",
                              INDENT,
                              fieldName);
                fieldType.createSerializer(
                    field, fieldName, INDENT1, output);
                output.format("%sretval |= 0x%x;%n",
                              INDENT1,
                              (1 << fieldIndex))
                      .format("%s}%n", INDENT);
            }
        }

        return;
    } // end of serializeFields(Formatter, List<>)

    /**
     * Generates the {@code booleanMask()} method.
     * @param retval add code to this output formatter.
     * @param fields message fields.
     */
    private static String serializeBooleans(final MessageType mt)
    {
        final List<MessageType.MessageField> fields =
            mt.fields();
        final int numFields = fields.size();
        int fieldIndex;
        MessageType.MessageField field;
        String fieldName;
        Class<?> dataClass;
        final Formatter retval = new Formatter();

        retval.format("private int booleanMask(%s value) {%n",
                      mt.dataClassName())
              .format("%sint retval = 0;%n%n", INDENT);

        for (fieldIndex = 0; fieldIndex< numFields; ++fieldIndex)
        {
            field = fields.get(fieldIndex);
            fieldName = String.format("value.%s", field.name());
            dataClass = (field.dataType()).dataClass();

            // Is this a boolean?
            if (isBoolean(dataClass))
            {
                retval.format("%sif (", INDENT);

                // Is this a primitive boolean?
                if (boolean.class.equals(dataClass))
                {
                    // Yes. Do not need to check for null.
                    retval.format("%s", fieldName);
                }
                // No. Then this is a Boolean object. Need to
                // check for null.
                else
                {
                    retval.format("%1$s != null && %1$s.equals(java.lang.Boolean.TRUE)",
                                  fieldName);
                }

                retval.format(") {%n")
                      .format("%sretval |= 0x%x;%n",
                              INDENT1,
                              (1 << fieldIndex))
                      .format("%s}%n%n", INDENT);
            }
        }

        retval.format("%sreturn (retval);%n", INDENT)
              .format("}");

        return (retval.toString());
    } // end of serializeBooleans(Formatter, List<>)

    /**
     * Generates code to deserialize all non-boolean fields.
     * @param output place code into this formatter.
     * @param fields all message fields.
     */
    private static void deserializeFields(final Formatter output,
                                          final List<MessageType.MessageField> fields)
    {
        final int numFields = fields.size();
        int fieldIndex;
        MessageType.MessageField field;
        String fieldName;
        DataType fieldType;

        for (fieldIndex = 0; fieldIndex < numFields; ++fieldIndex)
        {
            field = fields.get(fieldIndex);
            fieldName = field.name();
            fieldType = field.dataType();

            // Is this an object reference or a primitive type?
            // Skip boolean fields as well since they are already
            // "serialized" into the boolean mask.
            if (isBoolean(fieldType.dataClass()))
            {
                // no-op
            }
            else if (fieldType.isPrimitive() &&
                     !fieldType.isArray())
            {
                // A primitive type. Must appear in message.
                fieldType.createDeserializer(field,
                                             fieldName,
                                             INDENT,
                                             output,
                                             true);
            }
            // This is an object reference. Check the field mask
            // to see if it is null before de-serializing it.
            else
            {
                output.format("%sif ((fieldMask & 0x%x) == 0) {%n",
                              INDENT,
                              (1 << fieldIndex))
                      .format("%sbuilder.%s(null);%n",
                              INDENT1,
                              fieldName)
                      .format("%s} else {%n", INDENT);
                fieldType.createDeserializer(field,
                                             fieldName,
                                             INDENT1,
                                             output,
                                             true);
                output.format("%s}%n", INDENT);
            }
        }

        return;
    } // end of deserializeFields(Formatter, List<>)

    /**
     * Generates the code for converting the {@code boolMask}
     * integer into the boolean field settings.
     * @param output append code to this formatter.
     * @param fields message fields.
     */
    private static void deserializeBooleans(final Formatter output,
                                            final List<MessageType.MessageField> fields)
    {
        final int numFields = fields.size();
        int fieldIndex;
        MessageType.MessageField field;
        Class<?> dataClass;
        int fieldMask;

        for (fieldIndex = 0; fieldIndex < numFields; ++fieldIndex)
        {
            field = fields.get(fieldIndex);
            dataClass = (field.dataType()).dataClass();

            // Is this a boolean?
            if (isBoolean(dataClass))
            {
                // Yes.
                fieldMask = (1 << fieldIndex);

                // Is this a primitive boolean?
                if (boolean.class.equals(dataClass))
                {
                    // Yes.
                    output.format(
                        "%1$sbuilder.%2$s((boolMask & 0x%3$x) == 0x%3$x);%n",
                        INDENT,
                        field.name(),
                        fieldMask);
                }
                // No. Need to use a boolean object.
                else
                {
                    output.format(
                        "%1$sbuilder.%2$s(((boolMask & 0x%3$x) == 0x%3$x) ? java.lang.Boolean.TRUE : java.lang.Boolean.FALSE);%n",
                        INDENT,
                        field.name(),
                        fieldMask);
                }
            }
        }
        return;
    } // end of deserializeBooleans(Formatter, int, List<>)

    /**
     * Returns the {@code mc} instance constructed from the
     * generated {@code __TypeName__(List fields)} constructor
     * passing in {@code msgType.fields()} to the constructor.
     * @param msgType parent message type.
     * @param mc generated {@link MessageType} subclass.
     * @return {@code mc} instance.
     * @throws InstantiationException
     * if the instantiation fails. {@code InstantiationException}
     * cause is set to the under {@code Throwable} behind the
     * instantiation failure.
     */
    private static DataType instantiateMessageType(final ConcreteMessageType msgType,
                                                   final Class mc)
        throws InstantiationException
    {
        Object retval;

        try
        {
            final Object[] args =
            {
                msgType.fields(),
                msgType.replyTypes(),
                msgType.builder(),
                msgType.builderClass()
            };
            final MethodHandles.Lookup lookup =
                MethodHandles.publicLookup();
            final MethodType mt =
                MethodType.methodType(void.class,
                                      List.class,
                                      List.class,
                                      Method.class,
                                      Class.class);
            final MethodHandle ctor =
                lookup.findConstructor(mc, mt);

            // Need to put the msgType.fields() into a single
            // element array so the fields will be interpreted as
            // one argument. Otherwise, fields will be
            // interpreted as an argument list itself.
            retval = ctor.invokeWithArguments(args);
        }
        catch (Throwable tex)
        {
            final InstantiationException instex =
                new InstantiationException(
                    String.format(
                        "failed to instantiate %s class",
                        mc.getName()));

            instex.initCause(tex);

            throw (instex);
        }

        return ((DataType) retval);
    } // end of instantiateMessageType(ConcreteMessageType,Class)

    /**
     * Returns the reply classes associated with a request
     * message class via the {@link EReplyInfo} annotation. This
     * method is called by compiled request message class to
     * initialize the
     * {@code private static final Class[] REPLY_CLASSES} field.
     * @param args single element array containing the message
     * class name.
     * @return the requests associated reply message classes.
     * Returns an empty list for any error.
     */
    public static List<Class<? extends EReplyMessage>> replyClasses(final String[] args)
    {
        List<Class<? extends EReplyMessage>> retval =
            new ArrayList<>();

        if (args != null &&
            args.length == 1 &&
            args[0] != null &&
            !args[0].isEmpty())
        {
            try
            {
                @SuppressWarnings ("unchecked")
                final Class<? extends EMessageObject> mc =
                    (Class<? extends EMessageObject>)
                        Class.forName(args[0]);

                if ((ERequestMessage.class).isAssignableFrom(mc))
                {
                    // Get the reply classes associated with the
                    // request message class.
                    MessageType.replyClasses(mc, retval);
                }

            }
            catch (ClassNotFoundException |
                   InvalidMessageException jex)
            {
                // Ignore and return an empty list.
            }
        }

        return (Collections.unmodifiableList(retval));
    } // end of replyClasses(String)

    /**
     * Returns the character set with the given name. If
     * {@code name} is not valid, then returns {@link #CHARSET}.
     * @param name character set name.
     * @return named character set.
     */
    protected static Charset findCharset(final String name)
    {
        Charset retval;

        try
        {
            retval = Charset.forName(name);
        }
        catch (Exception jex)
        {
            retval = CHARSET;
        }

        return (retval);
    } // end of findCharset(String)
} // end of class DataType

//
// CHANGE LOG
// $Log: DataType.java,v $
// Revision 1.15  2008/01/19 14:37:28  charlesr
// Moved EAddress and Version to net.sf.eBus.messages package.
//
// Revision 1.14  2007/02/07 23:51:29  charlesr
// Changed deserialize return type back to Object.
//
// Revision 1.13  2006/09/01 14:09:39  charlesr
// Synchronized _dataTypeMap initialization.
//
// Revision 1.12  2006/08/20 12:56:25  charlesr
// Corrected comment.
//
// Revision 1.11  2006/07/19 23:21:46  charlesr
// Implemented Comparable interface.
//
// Revision 1.10  2006/07/02 19:25:46  charlesr
// Changed deserialize() to return a Comparable instead of
// an Object.
//
// Revision 1.9  2006/02/08 21:48:52  charlesr
// Added binary serialization by defining associated abstract
// methods.
//
// Revision 1.8  2006/01/10 19:02:42  charlesr
// Moved to Java 1.5.
//
// Revision 1.7  2005/03/11 21:21:34  charlesr
// Removed NameType as MessageName is no longer used.
//
// Revision 1.6  2005/03/07 18:19:39  charlesr
// Replace class.instanceOf() with class.equals().
//
// Revision 1.5  2005/02/28 19:27:13  charlesr
// Added EAddress data type.
//
// Revision 1.4  2005/02/27 17:50:42  charlesr
// + Added a default serialize() method which most subclasses now
//   use.
// + Moved _escape() and _unescape() methods to StringType
//   serialize() and deserialize() methods, respectively.
// + Limited eBus-supported data types to: boolean, byte, char,
//   date, double, float, integer, long, message name,
//   request ID, short, string and version.
//
// Revision 1.3  2004/12/14 18:11:46  charlesr
// Replaced notifyAll() with notify().
//
// Revision 1.2  2004/08/22 21:46:53  charlesr
// Added getDataClass() method. Improved javadoc comments.
//
// Revision 1.1  2004/08/15 21:11:45  charlesr
// Added MessageName and Version generic types to initial data
// types.
//
// Revision 1.0  2004/08/07 12:14:15  charlesr
// Initial revision
//
