// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright (C) 2019. Charles W. Rapp.
// All Rights Reserved.

package net.sf.eBus.messages.type;

import java.lang.invoke.MethodHandles;
import java.lang.invoke.MethodType;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.nio.BufferOverflowException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Formatter;
import java.util.List;
import net.sf.eBus.messages.EMessage;
import net.sf.eBus.messages.EMessageObject;
import net.sf.eBus.messages.ENotificationMessage;
import net.sf.eBus.messages.EReplyMessage;
import net.sf.eBus.messages.ERequestMessage;
import net.sf.eBus.messages.ESystemMessage;
import net.sf.eBus.messages.InvalidMessageException;
import net.sf.eBus.messages.ValidationException;
import static net.sf.eBus.messages.type.MessageType.PUBLIC_STATIC;
import static net.sf.eBus.messages.type.MessageType.findFields;

/**
 * Base class for concrete message types. While this class itself
 * is {@code abstract} it references a concrete message/field
 * type.
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public class ConcreteMessageType
    extends MessageType
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * Static builder method.
     */
    private final Method mBuilder;

    /**
     * Message builder class.
     */
    private final Class<? extends EMessageObject.Builder<?, ?>> mBuilderClass;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new concrete message type instance.
     * @param clazz the message class instance.
     * @param fields the message fields.
     * @param replies reply classes associated with a request
     * message class. Will be an empty array for non-request
     * message types.
     * @param builder message builder method.
     * @param bc message builder class used for
     * de-serialization.
     */
    protected ConcreteMessageType(final Class<?> clazz,
                                  final List<MessageField> fields,
                                  final List<Class<? extends EReplyMessage>> replies,
                                  final Method builder,
                                  final Class<? extends EMessageObject.Builder<?, ?>> bc)
    {
        super (clazz, fields, replies);

        mBuilder = builder;
        mBuilderClass = bc;
    } // end of ConcreteMessageType(...)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // DataType Abstract Method Overrides.
    //

    /**
     * Serializes the given {@link EMessageObject} to the given
     * buffer.
     * @param o serialize this object.
     * @param buffer contains serialized object.
     * @throws BufferOverflowException
     * if serialized {@code o} overflows {@code buffer}.
     */
    @Override
    public final void serialize(final Object o,
                                final ByteBuffer buffer)
        throws BufferOverflowException
    {
        // Is there a message object to serialize?
        if (o == null)
        {
            // No. There the field mask is zero - as in no fields
            // serialized.
            buffer.putInt(0);
        }
        // Yes, there is a message object to serialize. So get on
        // with it.
        else
        {
            final int fieldMaskPosition = buffer.position();

            // Problem: Setting the buffer's position beyond the
            // buffer's limit does not throw a
            // BufferOverflowException but an
            // IllegalArgumentException. So check if the buffer
            // has less than two bytes remaining.
            if (buffer.remaining() < FIELD_MASK_SIZE)
            {
                throw (new BufferOverflowException());
            }

            // Because we do not know which fields are null until
            // after we finish serializing the fields, move
            // past the field mask bytes and write the field mask
            // to this current position after serializing the
            // fields.
            buffer.position(fieldMaskPosition + FIELD_MASK_SIZE);

            // The payload is filled and we now have the field
            // mask. Go back and set the field mask.
            buffer.putInt(fieldMaskPosition,
                          serializeFields(o, buffer));
        }

        return;
    } // end of serialize(Object, ByteBuffer)

    /**
     * Returns an {@link EMessageObject} extracted from the given
     * buffer. This method expects the message subject to have
     * been previously set <em>if</em> the serialized object is
     * an {@link EMessage}.
     * @param buffer extract the serialized object from this
     * buffer.
     * @return the de-serialized object.
     * @throws BufferUnderflowException
     * if {@code buffer} contains an incomplete object
     * serialization.
     * @throws ValidationException
     * if the serialized object is invalid.
     */
    @Override
    public final Object deserialize(final ByteBuffer buffer)
        throws BufferUnderflowException,
               ValidationException
    {
        final int fieldMask = buffer.getInt();

        // Make sure subject(String) was called prior to
        // de-serializing.
        assert ((isMessage() && mSubject != null) ||
                (!isMessage() && mSubject == null)) :
                "subject incorrectly set for deserialization";

        return (deserializeFields(fieldMask, buffer));
    } // end of deserialize(ByteBuffer)

    /**
     * Returns code for serializing a specific field. This always
     * {@code UnsupportedOperationException} because this class
     * must be extended and this method overridden.
     * @param field message field.
     * @param fieldName generate serialization code for this
     * field.
     * @param indent indent code by this amount.
     * @param output place generated code into this formatter.
     * @throws UnsupportedOperationException
     * because this method must be overridden by a subclass.
     */
    @Override
    protected void createSerializer(final MessageField field,
                                    final String fieldName,
                                    final String indent,
                                    final Formatter output)
    {
        // Is this field an EMessage type?
        if (field.dataType().isMessage())
        {
            // Yes. Need to serialize the message subject.
            output.format(
                "%sSTRING_TYPE.serialize(((net.sf.eBus.messages.EMessage) %s).subject, buffer);%n",
                indent,
                fieldName);
        }

        // Now serialize the field itself.
        output.format(
            "%s(net.sf.eBus.messages.type.DataType.findType(%s.class)).serialize(%s, buffer);%n",
            indent,
            (field.javaType()).getCanonicalName(),
            fieldName);

        return;
    } // end of createSerializer(String, String, Formatter)

    /**
     * Returns code for de-serializing a specified field. This
     * always {@code UnsupportedOperationException} because this
     * class must be extended and this method overridden.
     * @param field message field.
     * @param fieldName generate de-serialization code for this
     * field.
     * @param indent indent code by this amount.
     * @param output place generated code into this formatter.
     * @param useBuilder if {@code true} store de-serialized
     * object in a builder; otherwise store in local variable
     * named {@code fieldName}.
     * @throws UnsupportedOperationException
     * because this method must be overridden by a subclass.
     */
    @Override
    protected void createDeserializer(final MessageField field,
                                      final String fieldName,
                                      final String indent,
                                      final Formatter output,
                                      final boolean useBuilder)
    {
        output.format(
            "%snet.sf.eBus.messages.type.MessageType mt = (net.sf.eBus.messages.type.MessageType) net.sf.eBus.messages.type.DataType.findType(%s.class);%n%n",
            indent,
            (field.javaType()).getCanonicalName());

        // Is this field an EMessage type?
        if (field.dataType().isMessage())
        {
            // Yes. Need to de-serialize the message subject and
            // store it away in the
            output.format(
                "%smt.subject((java.lang.String) STRING_TYPE.serialize(((net.sf.eBus.messages.EMessage) %s).subject, buffer));%n",
                indent,
                fieldName);
        }

        // Now de-serialize the message fields.
        final String format =
            (useBuilder ?
             "%sbuilder.%s((%s) mt.deserialize(buffer));%n" :
             "%s%s = (%s) mt.deserialize(buffer);%n");
        String fieldType = (field.javaType()).getCanonicalName();

        // Note: arrays are deserialized one element at a time.
        // The field name is "arrayValue[i]" and useBuilder is
        // false. BUT fieldType is like "int[]" - which is
        // incorrect. It should be "int" because only one int
        // is being deserialized. So if this is an array field,
        // remove the "[]" from the end of the field type.
        if (field.isArray())
        {
            fieldType =
                fieldType.substring(0, (fieldType.length() - 2));
        }

        output.format(format,
                      indent,
                      fieldName,
                      fieldType);

        return;
    } // end of createDeserializer(...)

    //
    // end of DataType Abstract Method Overrides.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get Methods.
    //

    /**
     * Returns the static method which returns the message
     * builder instance.
     * @return builder static method.
     */
    public Method builder()
    {
        return (mBuilder);
    } // end of builder()

    /**
     * Returns the message object builder class.
     * @return message object builder class.
     */
    public Class<? extends EMessageObject.Builder<?, ?>> builderClass()
    {
        return (mBuilderClass);
    } // end of builderClass()

    /**
     * Returns the message builder canonical name.
     * @return message builder canonical name.
     */
    public String builderClassName()
    {
        return (mBuilderClass.getCanonicalName());
    } // end of builderClassName()

    //
    // end of Get Methods.
    //-----------------------------------------------------------

    /**
     * Returns the concrete message type instance for the
     * specified message or field class.
     * @param mc the message or field class.
     * @return the message type used to serialize, de-serialize
     * the message class.
     * @throws InvalidMessageException
     * if {@code mc} is an invalid message class.
     */
    /* package */ static ConcreteMessageType load(final Class<? extends EMessageObject> mc)
            throws InvalidMessageException
    {
        final List<MessageField> fields = findFields(mc);
        final List<Class<? extends EReplyMessage>> replyClasses =
            new ArrayList<>();
        Method builder;
        Class<? extends EMessageObject.Builder<?, ?>> bc;

        try
        {
            // Look up the message builder and then the builder
            // return type.
            builder = findBuilder(mc);
            bc = findBuilderClass(mc, builder, fields);
        }
        catch (NoSuchMethodException methex)
        {
            throw (
                new InvalidMessageException(
                    mc,
                    methex.getMessage(),
                    methex));
        }

        // Is this a request message type?
        if ((ERequestMessage.class).isAssignableFrom(mc))
        {
            // Yes, this is a request message type. Retrieve the
            // reply message classes from the @ReplyInfo class
            // annotation.
            replyClasses(mc, replyClasses);
        }

        // Now that the builder method is found and validated,
        // the builder setter method for each field can be set.
        setSetters(bc, fields);

        return (
            new ConcreteMessageType(
                mc,
                Collections.unmodifiableList(fields),
                Collections.unmodifiableList(replyClasses),
                builder,
                bc));
    } // end of load(Class)

    private static Method findBuilder(final Class<? extends EMessageObject> mc)
        throws NoSuchMethodException
    {
        final Method retval = mc.getMethod(BUILDER_METHOD);

        // Is the builder method "public static"?
        if ((retval.getModifiers() & PUBLIC_STATIC) != PUBLIC_STATIC)
        {
            throw (
                new NoSuchMethodException(
                    BUILDER_METHOD + " method is not public final"));
        }

        return (retval);
    }

    /**
     * Returns the builder class associated with this
     * {@code EMessageObject} subclass where {@code mc} is
     * required to have a {@code builder()} method.
     * @param mc find builder for this message class.
     * @param fields message fields list.
     * @return builder class.
     * @throws NoSuchMethodException
     * if {@code mc} does not have the required {@code builder()}
     * method.
     */
    @SuppressWarnings ("unchecked")
    private static Class<? extends EMessageObject.Builder<?, ?>> findBuilderClass(final Class<? extends EMessageObject> mc,
                                                                                  final Method builder,
                                                                                  final List<MessageField> fields)
        throws NoSuchMethodException
    {
        final Class<? extends EMessageObject.Builder<?, ?>> retval =
            (Class<? extends EMessageObject.Builder<?, ?>>) builder.getReturnType();
        final Class<?> baseBuilder = baseBuilder(mc);
        final int modifiers = retval.getModifiers();

        // Does the return type extend the correct builder
        // base class?
        if (!baseBuilder.isAssignableFrom(retval))
        {
            throw (
                new NoSuchMethodException(
                     retval.getCanonicalName() + " does not extend " + baseBuilder.getCanonicalName()));
        }

        // Is the builder class public and static?
        if ((modifiers & PUBLIC_STATIC) != PUBLIC_STATIC)
        {
            throw (
                new NoSuchMethodException(
                    retval.getCanonicalName() + " is not public"));
        }

        // Verify that the build has set methods which match the
        // name and type of the message class fields.
        checkSetters(retval, fields);

        return (retval);
    } // end of findBuilderClass(Class, List<>)

    /**
     * Returns the builder base class matching the message object
     * class.
     * @param mc message class.
     * @return base builder for the given message class.
     */
    @SuppressWarnings ("unchecked")
    private static Class<? extends EMessageObject.Builder<?, ?>> baseBuilder(final Class<? extends EMessageObject> mc)
    {
        Class<?> retval;

        if (ENotificationMessage.class.isAssignableFrom(mc))
        {
            retval = ENotificationMessage.Builder.class;
        }
        else if (ERequestMessage.class.isAssignableFrom(mc))
        {
            retval = ERequestMessage.Builder.class;
        }
        else if (EReplyMessage.class.isAssignableFrom(mc))
        {
            retval = EReplyMessage.Builder.class;
        }
        else if (ESystemMessage.class.isAssignableFrom(mc))
        {
            retval = ESystemMessage.Builder.class;
        }
        else
        {
            retval = EMessageObject.Builder.class;
        }

        return ((Class<? extends EMessageObject.Builder<?, ?>>) retval);
    } // end of baseBuilder(Class)

    /**
     * Verify that the builder class {@code bc} has a correctly
     * declared setter method for each of the message fields.
     * @param fields public message fields.
     * @param bc builder class.
     * @throws NoSuchMethodException
     * if {@code bc} is missing a required setter method or
     * the setter method is incorrectly declared.
     */
    private static void checkSetters(final Class<?> bc,
                                     final List<MessageField> fields)
        throws NoSuchMethodException
    {
        String methodName = "(not set)";
        Method m;
        Class<?> rt;

        for (MessageField mf : fields)
        {
            try
            {
                methodName = mf.name();
                m = bc.getMethod(methodName,
                                 (mf.dataType()).dataClass());
                rt = m.getReturnType();

                // Is this setter method public?
                if ((m.getModifiers() & Modifier.PUBLIC) == 0)
                {
                    throw (
                        new NoSuchMethodException(
                            methodName + " not public"));
                }

                // Does this setter method return self?
                if (!rt.isAssignableFrom(bc))
                {
                    throw (
                        new NoSuchMethodException(
                            methodName +
                            " return type " +
                            rt.getCanonicalName() +
                            " not " +
                            bc.getCanonicalName()));
                }
            }
            catch (NoSuchMethodException |
                   SecurityException secex)
            {
                throw (
                    new NoSuchMethodException(
                        bc.getCanonicalName() +
                        " does not have a setter method for " +
                        methodName));
            }
        }

        return;
    } // end of checkSetters(Class, List<>)

    /**
     * Sets each field's matching builder setter method handle.
     * This must be done after field construction because the
     * fields must first be loaded so the message builder class
     * can be found and validated. After that, the message field
     * setters can be loaded.
     * @param bc builder class.
     * @param fields message object fields.
     */
    private static void setSetters(final Class<? extends EMessageObject.Builder<?, ?>> bc,
                                   final List<MessageField> fields)
    {
        final MethodHandles.Lookup lookup =
            MethodHandles.publicLookup();
        MethodType setterType;

        for (MessageField mf : fields)
        {
            setterType =
                MethodType.methodType(
                    bc, (mf.dataType()).dataClass());

            try
            {
                mf.setter(
                    lookup.findVirtual(
                        bc, mf.name(), setterType));
            }
            catch (IllegalAccessException | NoSuchMethodException jex)
            {
                // The builder method is already known to exist
                // and has the correct return and parameter
                // types. That means these exceptions will not be
                // thrown.
            }
        }

        return;
    } // end of setSetters(Class, List<>)
} // end of class ConcreteMessageType
