//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2013, 2014, 2019. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.messages.type;

import java.nio.BufferOverflowException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.util.Formatter;
import net.sf.eBus.messages.type.MessageType.MessageField;

/**
 * This class is able to serialize and de-serialize a
 * {@link java.lang.Class class} instance so that it may be sent
 * via an {@link net.sf.eBus.messages.EMessage eBus message}.
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public final class ClassType
    extends DataType
{
//---------------------------------------------------------------
// Member data.
//

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new {@link java.lang.Class} data type
     * serializer. The serialized size is variable.
     */
    /* package */ ClassType()
    {
        super (Class.class, true, VARIABLE_SIZE, null);
    } // end of ClassType()

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // DataType Abstract Method Overrides.
    //

    /**
     * Serializes the class instance to the give buffer. Writes
     * nothing if {@code o} is {@code null} or not a
     * {@code java.lang.Class} instance.
     * @param o serialize this class instance.
     * @param buffer write to this buffer.
     * @throws BufferOverflowException
     * if {@code buffer} does not contain sufficient space to
     * hold the class name.
     */
    @Override
    public void serialize(final Object o,
                          final ByteBuffer buffer)
        throws BufferOverflowException
    {
        if (o instanceof Class)
        {
            STRING_TYPE.serialize(((Class) o).getName(), buffer);
        }

        return;
    } // end of serialize(Object, ByteBuffer)

    /**
     * Returns the class instance de-serialized from the given
     * buffer. May return {@code null}.
     * @param buffer read in the class instance from this buffer.
     * @return the class object.
     * @throws BufferUnderflowException
     * if {@code buffer} contains an incomplete class name.
     */
    @Override
    public Object deserialize(final ByteBuffer buffer)
        throws BufferUnderflowException
    {
        final String name =
            (String) STRING_TYPE.deserialize(buffer);
        Class<?> retval = null;

        if (name != null && !name.isEmpty())
        {
            retval = sClasses.get(name);
            if (retval == null)
            {
                try
                {
                    retval = Class.forName(name);
                    sClasses.put(name, retval);
                }
                catch (ClassNotFoundException classex)
                {
                    // Ignore and return null.
                }
            }
        }

        return (retval);
    } // end of deserialize(ByteBuffer)

    /**
     * Adds the Java code used to serialize the Class instance to
     * a {@link ByteBuffer}.
     * @param field message field.
     * @param fieldName the message field name.
     * @param indent indent the code by this amount.
     * @param output write the code to this formatter.
     */
    @Override
    protected void createSerializer(final MessageField field,
                                    final String fieldName,
                                    final String indent,
                                    final Formatter output)
    {
        output.format(
            "%sfinal String name = (%s).getName();%n",
            indent,
            fieldName);
        STRING_TYPE.createSerializer(
            field, "name", indent, output);

        return;
    } // end of createSerializer(...)

    /**
     * Generates the class de-serialization code storing the
     * result according to the {@code useBuilder} flag. If
     * {@code true}, then store the result into the message/field
     * builder. Otherwise store the result into the named local
     * variable.
     * @param field message field.
     * @param fieldName store result into this field.
     * @param indent indent the code by this amount.
     * @param output append the code to this formatter.
     * @param useBuilder if {@code true} store de-serialized
     * class into the message/field builder; if {@code false}
     * store result into named local variable.
     */
    @Override
    protected void createDeserializer(final MessageField field,
                                      final String fieldName,
                                      final String indent,
                                      final Formatter output,
                                      final boolean useBuilder)
    {
        final String indent1 = indent + INDENT;
        final String indent2 = indent1 + INDENT;
        final String indent3 = indent2 + INDENT;

        output.format("%sjava.lang.String _name;%n", indent);
        STRING_TYPE.createDeserializer(
            field, "_name", indent, output, false);
        output.format("%sif (_name != null && !_name.isEmpty()) {%n",
                      indent)
              .format("%s java.lang.Class _clazz = (java.lang.Class) sClasses.get(_name);%n",
                      indent1)
              .format("%sif (_clazz == null) {%n", indent1)
              .format("%stry {%n", indent2)
              .format("%s_clazz = Class.forName(_name);%n", indent3)
              .format("%ssClasses.put(_name, _clazz);%n", indent3)
              .format("%s} catch (ClassNotFoundException classex) {%n",
                      indent2)
              .format("%s_clazz = null;%n", indent3)
              .format("%s}%n", indent2)
              .format("%s}%n", indent1)
              .format((useBuilder ?
                       "%sbuilder.%s(_clazz);%n" :
                       "%s%s = _clazz;%n"),
                      indent1,
                      fieldName)
              .format("%s}%n", indent);

        return;
    } // end of createDeserializer(...)

    //
    // end of DataType Abstract Method Overrides.
    //-----------------------------------------------------------
} // end of class ClassType
