//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright (C) 2005 - 2008, 2011 - 2014, 2019. Charles W. Rapp.
// All Rights Reserved.
//

package net.sf.eBus.messages.type;

import java.nio.BufferOverflowException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.util.Formatter;
import net.sf.eBus.messages.type.MessageType.MessageField;

/**
 * Provides binary serialization/de-serialization of 2-byte
 * character values.
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public final class CharType
    extends DataType
{
//---------------------------------------------------------------
// Member data.
//

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a single character object. Associates this data
     * type with the SQL char type.
     * @param jType char class.
     */
    /* package */ CharType(final Class<?> jType)
    {
        super (jType, true, Character.BYTES, (char) 0);
    } // end of CharType(Class)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // DataType Abstract Methods Overrides.
    //

    /**
     * Serializes this {@code java.lang.Character} to the given
     * {@code buffer}.
     * @param o a {@code java.lang.Character} object.
     * @param buffer serialize to this buffer.
     * @exception BufferOverflowException
     * if {@code buffer} overflows.
     * @see #deserialize(ByteBuffer)
     */
    @Override
    public void serialize(final Object o, final ByteBuffer buffer)
        throws BufferOverflowException
    {
        if (o instanceof Character)
        {
            buffer.putChar((Character) o);
        }

        return;
    } // end of serialize(Object, ByteBuffer)

    /**
     * Returns the deserialized {@code java.lang.Character} in
     * {@code buffer}.
     * @param buffer deserialize the object from this
     * {@code ByteBuffer}.
     * @return the deserialized {@code java.lang.Character}.
     * @exception BufferUnderflowException
     * if {@code buffer} contains an incomplete character.
     * @see #serialize(Object, ByteBuffer)
     */
    @Override
    public Object deserialize(final ByteBuffer buffer)
        throws BufferUnderflowException
    {
        return (buffer.getChar());
    } // end of deserialize(ByteBuffer)

    /**
     * Adds the Java code used to serialize the character field
     * to a {@link ByteBuffer}.
     * @param field message field.
     * @param fieldName the message field name.
     * @param indent indent the code by this amount.
     * @param output write the code to this formatter.
     */
    @Override
    protected void createSerializer(final MessageField field,
                                    final String fieldName,
                                    final String indent,
                                    final Formatter output)
    {
        output.format(
            "%sbuffer.putChar(%s);%n", indent, fieldName);

        return;
    } // end of createSerializer(...)

    /**
     * Generates the code to de-serialize the
     * from a {@link ByteBuffer}.
     * @param field message field.
     * @param fieldName store the de-serialized field in this
     * name.
     * @param indent indent the code by this amount.
     * @param output append the code to this formatter.
     * @param useBuilder if {@code true} then {@code fieldName}
     * is a builder method name; otherwise a local variable.
     */
    @Override
    protected void createDeserializer(final MessageField field,
                                      final String fieldName,
                                      final String indent,
                                      final Formatter output,
                                      final boolean useBuilder)
    {
        final String format =
            (useBuilder ?
             "%sbuilder.%s(buffer.getChar());%n" :
             "%s%s = buffer.getChar();%n");

        output.format(format, indent, fieldName);

        return;
    } // end of createDeserializer(...)

    //
    // end of DataType Abstract Methods Overrides.
    //-----------------------------------------------------------
} // end of CharType
