//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright (C) 2004 - 2008, 2011 - 2014, 2019. Charles W. Rapp.
// All Rights Reserved.
//

package net.sf.eBus.messages.type;

import java.nio.BufferOverflowException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.util.Formatter;
import net.sf.eBus.messages.type.MessageType.MessageField;

/**
 * Provides binary serialization/de-serialization of boolean
 * values.
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public final class BooleanType
    extends DataType
{
//---------------------------------------------------------------
// Member data.
//

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates the boolean type object for the given class.
     * As of eBus 5.2.0, boolean variables are represented as one
     * bit, hence a zero byte size.
     * @param jType the Java class.
     */
    /* package */ BooleanType(final Class<?> jType)
    {
        super (jType, true, 0, Boolean.FALSE);
    } // end of BooleanType(Class)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // DataType Abstract Method Overrides.
    //

    /**
     * Serializes this boolean to the given {@code buffer}.
     * @param o a {@code Boolean} object. May be {@code null}.
     * @param buffer write to this buffer.
     * @exception BufferOverflowException
     * if {@code buffer} overflows.
     * @see #deserialize(ByteBuffer)
     */
    @Override
    public void serialize(final Object o,
                          final ByteBuffer buffer)
        throws BufferOverflowException
    {
        if (o instanceof Boolean)
        {
            final boolean bool = (Boolean) o;
            final byte b = (bool ? (byte) 1 : (byte) 0);

            buffer.put(b);
        }

        return;
    } // end of serialize(Object, ByteBuffer)

    /**
     * Returns the deserialized {@code Boolean} in
     * {@code buffer}.
     * @param buffer Deserialize the boolean object from this
     * {@code ByteBuffer}.
     * @return the deserialized {@code Boolean}.
     * @exception BufferUnderflowException
     * if {@code buffer} contains fewer bytes than required for
     * a boolean type.
     *
     * @see #serialize(java.lang.Object, java.nio.ByteBuffer)
     */
    @Override
    public Object deserialize(final ByteBuffer buffer)
        throws BufferUnderflowException
    {
        return (buffer.get() == (byte) 1 ?
                    Boolean.TRUE :
                    Boolean.FALSE);
    } // end of deserialize(ByteBuffer)

    /**
     * Boolean values are serialized separately into a bit mask.
     * No code is generated.
     * @param field message field.
     * @param fieldName the message field name.
     * @param indent indent the code by this amount.
     * @param output write the code to this formatter.
     */
    @Override
    protected void createSerializer(final MessageField field,
                                    final String fieldName,
                                    final String indent,
                                    final Formatter output)
    {
        // No code generated since boolean fields are handled
        // separately.
    } // end of createSerializer(...)

    /**
     * Boolean values are serialized separately into a bit mask.
     * No code is generated.
     * @param field message field.
     * @param fieldName store the de-serialized field in this
     * name.
     * @param indent indent the code by this amount.
     * @param output append the code to this formatter.
     * @param useBuilder if {@code true} then {@code fieldName}
     * is a builder method name; otherwise a local variable.
     */
    @Override
    protected void createDeserializer(final MessageField field,
                                      final String fieldName,
                                      final String indent,
                                      final Formatter output,
                                      final boolean useBuilder)
    {
        // No code generated since boolean fields are handled
        // separately.
    } // end of createDeserializer(...)

    //
    // end of DataType Abstract Method Overrides.
    //-----------------------------------------------------------
} // end of class BooleanType
