//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2017, 2019. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.messages.type;

import java.math.BigInteger;
import java.nio.BufferOverflowException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.util.Formatter;
import net.sf.eBus.messages.type.MessageType.MessageField;

/**
 * Used to serialize and de-serialize a {@link BigInteger}
 * instance using the {@code byte[]} returned by
 * {@link BigInteger#toByteArray()}.
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public final class BigIntegerType
    extends DataType
{
//---------------------------------------------------------------
// Member data.
//

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new instance of BigIntegerType.
     * {@link BigInteger} values of of variable size.
     */
    /* package */ BigIntegerType()
    {
        super (BigInteger.class, true, VARIABLE_SIZE, null);
    } // end of BigIntegerType()

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // DataType Abstract Method Overrides.
    //

    /**
     * Serializes the given {@code BigInteger} two's-complement
     * byte array to the given buffer.
     * @param o serialize this object.
     * @param buffer serialize {@code o} to this buffer.
     * @throws BufferOverflowException
     * if the serialized {@code BigInteger} instance is greater
     * than the buffer's remaining space.
     */
    @Override
    public void serialize(final Object o,
                          final ByteBuffer buffer)
        throws BufferOverflowException
    {
        if (o instanceof BigInteger)
        {
            final byte[] b = ((BigInteger) o).toByteArray();

            buffer.putShort((short) b.length);
            buffer.put(b);
        }

        return;
    } // end of serialize(Object, ByteBuffer)

    /**
     * De-serializes the {@code BigInteger} two's-complement
     * byte array from the given buffer and constructs a
     * {@code BigInteger} instance from that array. If the
     * de-serialized byte array is of zero length, then returns
     * {@code null}.
     * @param buffer read in the {@code BigInteger} from this
     * buffer.
     * @return the de-serialized {@code BigInteger} instance.
     * May return {@code null}.
     * @throws BufferUnderflowException
     * if {@code buffer} contains an incomplete
     * {@code BigInteger} instance.
     */
    @Override
    public Object deserialize(final ByteBuffer buffer)
        throws BufferUnderflowException
    {
        final int size = buffer.getShort();
        BigInteger retval = null;

        if (size > 0)
        {
            final byte[] b = new byte[size];

            buffer.get(b);
            retval = new BigInteger(b);
        }

        return (retval);
    } // end of deserialize(ByteBuffer)

    /**
     * Generates runtime-compilable code for serializing a
     * {@code BigInteger} message field.
     * @param field message field.
     * @param fieldName {@code BigInteger} field name.
     * @param indent indent generated code lines by this amount.
     * @param output write generated code to this output.
     */
    @Override
    protected void createSerializer(final MessageField field,
                                    final String fieldName,
                                    final String indent,
                                    final Formatter output)
    {
        final String indent1 = indent + INDENT;

        output.format(
            "%sfinal byte[] bytes = (%s).toByteArray();%n",
            indent,
            fieldName);
        output.format(
            "%sbuffer.putShort((short) bytes.length);%n",
            indent);
        output.format("%sif (bytes.length > 0) {%n", indent);
        output.format("%sbuffer.put(bytes);%n", indent1);
        output.format("%s}%n", indent);

        return;
    } // end of createSerializer(String, String, Formatter)

    /**
     * Generates runtime-compilable code for de-serializing a
     * {@code BigInteger} message field from a
     * {@link ByteBuffer}.
     * @param field message field.
     * @param fieldName store de-serialized {@code BigInteger}
     * instance into this field.
     * @param indent indent generated code lines by this amount.
     * @param output write generated code to this output.
     * @param useBuilder if {@code true} then {@code fieldName}
     * is a builder method name; otherwise a local variable.
     */
    @Override
    protected void createDeserializer(final MessageField field,
                                      final String fieldName,
                                      final String indent,
                                      final Formatter output,
                                      final boolean useBuilder)
    {
        final String indent1 = indent + INDENT;
        final String format =
            (useBuilder ?
             "%sbuilder.%s(new java.math.BigInteger(b));%n" :
             "%s%s = new java.math.BigInteger(b);%n");

        output.format("%sfinal int size = buffer.getShort();%n",
                      indent)
              .format("%sif (size == 0) {%n", indent)
              .format("%sbuilder.%s(null);%n",
                      indent1,
                      fieldName)
              .format("%s} else {%n", indent)
              .format("%sfinal byte[] b = new byte[size];%n",
                      indent1)
              .format("%sbuffer.get(b);%n", indent1)
              .format(format, indent1, fieldName)
              .format("%s}%n", indent);

        return;
    } // end of createDeserializer(String, String, Formatter)

    //
    // end of DataType Abstract Method Overrides.
    //-----------------------------------------------------------
} // end of class BigIntegerType
