//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright (C) 2013, 2014, 2019. Charles W. Rapp.
// All Rights Reserved.
//

package net.sf.eBus.messages.type;

import java.math.BigDecimal;
import java.nio.BufferOverflowException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.util.Formatter;
import net.sf.eBus.messages.type.MessageType.MessageField;

/**
 * Used to explicity serialize and de-serialize a
 * {@link BigDecimal} instance using the {@code long} value and
 * {@code int} scale.
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public final class BigDecimalType
    extends DataType
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * A {@link BigDecimal} requires {@value} bytes for encoding.
     */
    public static final int BYTES = 12;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new instance of BigDecimalType.
     */
    /* package */ BigDecimalType()
    {
        super (BigDecimal.class, true, BYTES, null);
    } // end of BigDecimalType()

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // DataType Abstract Method Overrides.
    //

    @Override
    public void serialize(final Object o,
                          final ByteBuffer buffer)
        throws BufferOverflowException
    {
        if (o instanceof BigDecimal)
        {
            final BigDecimal bd = (BigDecimal) o;

            buffer.putLong((bd.unscaledValue()).longValue())
                  .putInt(bd.scale());
        }

        return;
    } // end of serialize(Object, ByteBuffer)

    @Override
    public Object deserialize(final ByteBuffer buffer)
        throws BufferUnderflowException
    {
        return (
            BigDecimal.valueOf(
                buffer.getLong(), buffer.getInt()));
    } // end of deserialize(ByteBuffer)

    /**
     * Generates the code to serialize the {@code BigDecimal}
     * instance to a {@link ByteBuffer}.
     * @param field message field.
     * @param fieldName the {@code BigDecimal} field.
     * @param indent indent the code by this amount.
     * @param output append the code to this formatter.
     */
    @Override
    protected void createSerializer(final MessageField field,
                                    final String fieldName,
                                    final String indent,
                                    final Formatter output)
    {
        output.format(
            "%sbuffer.putLong(((%s).unscaledValue()).longValue())%n",
            indent,
            fieldName);
        output.format(
            "%s      .putInt((%s).scale());%n",
            indent,
            fieldName);

        return;
    } // end of createSerializer(...)

    /**
     * Generates the code to de-serialize the {@code BigDecimal}
     * instance from a {@link ByteBuffer}.
     * @param field message field.
     * @param fieldName store the de-serialized field in this
     * name.
     * @param indent indent the code by this amount.
     * @param output append the code to this formatter.
     * @param useBuilder if {@code true} then {@code fieldName}
     * is a builder method name; otherwise a local variable.
     */
    @Override
    protected void createDeserializer(final MessageField field,
                                      final String fieldName,
                                      final String indent,
                                      final Formatter output,
                                      final boolean useBuilder)
    {
        final String format =
            (useBuilder ?
             "%sbuilder.%s(java.math.BigDecimal.valueOf(buffer.getLong(), buffer.getInt()));%n" :
             "%s%s = java.math.BigDecimal.valueOf(buffer.getLong(), buffer.getInt()));%n");

        output.format(format, indent, fieldName);

        return;
    } // end of createDeserializer(...)

    //
    // end of DataType Abstract Method Overrides.
    //-----------------------------------------------------------
} // end of class BigDecimalType
