//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright (C) 2007, 2008, 2011 - 2014, 2019. Charles W. Rapp.
// All Rights Reserved.
//

package net.sf.eBus.messages.type;

import java.lang.reflect.Array;
import java.nio.BufferOverflowException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.util.Formatter;
import net.sf.eBus.messages.type.MessageType.MessageField;

/**
 * Provides binary serialization/de-serialization of an object
 * collection. The collection must contain
 * {@link net.sf.eBus.messages.EMessageObject} objects of the
 * same class type.
 * <p>
 * Supports array sizes up to {@link #MAX_ARRAY_SIZE} items.
 * </p>
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public final class ArrayType
    extends DataType
{
//---------------------------------------------------------------
// Member data.
//

    /**
     * The list consists of this data type.
     */
    private final DataType mSubtype;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates an array type which encapsulates the specified
     * data type. Arrays are homogenous collections and
     * considered a built-in type.
     * @param jClass the Java array class.
     * @param subtype the array element type.
     */
    public ArrayType(final Class<?> jClass,
                     final DataType subtype)
    {
        super (jClass, true, VARIABLE_SIZE, null);

        mSubtype = subtype;
    } // end of ArrayType(DataType)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // DataType Abstract Method Overrides.
    //

    /**
     * Returns {@code true} if {@code o} is either {@code null}
     * or a {@code Collection} instance; {@code false} otherwise.
     * @param o Check this object's type.
     * @return {@code true} if {@code o} is either {@code null}
     * or a {@code Collection} instance; {@code false} otherwise.
     */
    @Override
    public boolean isInstance(final Object o)
    {
        return (o == null || mClass.isInstance(o));
    } // end of isInstance(Object)

    /**
     * Serializes this list to the buffer.
     * @param o a list of objects.
     * @param buffer serialize to this buffer.
     * @exception BufferOverflowException
     * if {@code buffer} overflows.
     * @see #deserialize(ByteBuffer)
     */
    @Override
    public void serialize(final Object o, final ByteBuffer buffer)
        throws BufferOverflowException
    {
        if (o != null)
        {
            final int size = Array.getLength(o);
            int index;

            if (size > MAX_ARRAY_SIZE)
            {
                final BufferOverflowException bufex =
                    new BufferOverflowException();

                bufex.initCause(
                    new IllegalArgumentException(
                        String.format(
                            "array length %,d exceeds max allowed size %,d",
                            size,
                            MAX_ARRAY_SIZE)));

                throw (bufex);
            }

            // First output the array size.
            buffer.putShort((short) size);

            // Then use the subtype to output the items.
            if (byte.class.equals(mSubtype.dataClass()))
            {
                buffer.put((byte[]) o);
            }
            else
            {
                for (index = 0; index < size; ++index)
                {
                    mSubtype.serialize(
                        Array.get(o, index), buffer);
                }
            }
        }

        return;
    } // end of serialize(Object, ByteBuffer)

    /**
     * Returns the deserialized collection as an {@code Object}.
     * If the array size is zero, then returns an empty
     * collection. The returned value may be safely typecast to
     * {@code java.util.Collection}.
     * @param buffer Deserialize the list object from this
     * {@code ByteBuffer}.
     * @return the deserialized {@code Collection}.
     * @exception BufferUnderflowException
     * if {@code buffer} contains an incomplete serialized array.
     * @see #serialize(Object, ByteBuffer)
     */
    @Override
    @SuppressWarnings("unchecked")
    public Object deserialize(final ByteBuffer buffer)
        throws BufferUnderflowException
    {
        final int size = buffer.getShort();
        Object retval = null;

        if (size < 0 || size > MAX_ARRAY_SIZE)
        {
            final BufferUnderflowException bufex =
                new BufferUnderflowException();

            bufex.initCause(
                new IllegalArgumentException(
                    "invalid array size " + size));

            throw (bufex);
        }
        else if (byte.class.equals(mSubtype.dataClass()))
        {
            retval = new byte[size];
            buffer.get((byte[]) retval);
        }
        else
        {
            int index;

            retval =
                Array.newInstance(mSubtype.dataClass(), size);

            for (index = 0; index < size; ++index)
            {
                Array.set(retval,
                          index,
                          mSubtype.deserialize(buffer));
            }
        }

        return (retval);
    } // end of deserialize(ByteBuffer)

    /**
     * Generates the array serialization code.
     * @param field message field.
     * @param fieldName fully-qualified field name.
     * @param indent indent generated code by this spacing.
     * @param output place generated code into this formatter.
     */
    @Override
    protected void createSerializer(final MessageField field,
                                    final String fieldName,
                                    final String indent,
                                    final Formatter output)
    {
        final String indent1 = indent + INDENT;

        output.format(
            "%sif ((%s).length > MAX_ARRAY_SIZE) {%n",
            indent,
            fieldName);
        output.format(
            "%sfinal java.nio.BufferOverflowException bufex = new java.nio.BufferOverflowException();%n",
            indent1);
        output.format(
            "%sbufex.initCause(new IllegalArgumentException(\"array length exceeds MAX_ARRAY_SIZE\"));%n",
            indent1);
        output.format("%sthrow (bufex);%n", indent1);
        output.format("%s}%n", indent);
        output.format(
            "%sbuffer.putShort((short) (%s).length);%n",
            indent,
            fieldName);

        // If the subtype is byte, then have ByteBuffer write out
        // the byte array in one go.
        if ((byte.class).equals(mSubtype.dataClass()))
        {
            output.format(
                "%sbuffer.put(%s);%n", indent, fieldName);
        }
        else
        {
            final String subtypeName =
                String.format("%s[index]", fieldName);

            output.format(
                "%sfor (int index = 0; index < (%s).length; ++index) {%n",
                indent,
                fieldName);
            mSubtype.createSerializer(
                field, subtypeName, indent1, output);
            output.format("%s}%n", indent);
        }

        return;
    } // end of createSerializer(...)

    /**
     * Generates the Java code for de-serializing an array.
     * @param field message field.
     * @param fieldName the array message field name.
     * @param indent indent the code by this amount.
     * @param output append the code to this formatter.
     * @param useBuilder if {@code true} then {@code fieldName}
     * is a builder method name; otherwise a local variable.
     */
    @Override
    protected void createDeserializer(final MessageField field,
                                      final String fieldName,
                                      final String indent,
                                      final Formatter output,
                                      final boolean useBuilder)
    {
        final String indent1 = indent + INDENT;

        output.format("%sfinal int arraySize = (int) buffer.getShort();%n",
                      indent)
              .format("%sif (arraySize < 0 || arraySize > MAX_ARRAY_SIZE) {%n",
                      indent)
              .format("%sfinal java.nio.BufferUnderflowException bufex = new java.nio.BufferUnderflowException();%n",
                      indent1)
              .format("%sbufex.initCause(new IllegalArgumentException(\"invalid array size\"));%n",
                      indent1)
              .format("%sthrow (bufex);%n", indent1)
              .format("%s}%n", indent)
              .format("%1$sfinal %2$s[] arrayValue = new %2$s[arraySize];%n",
                      indent,
                      mSubtype.dataClassName());

        // If the subtype is byte, then have ByteBuffer read in
        // the byte array in one go.
        if ((byte.class).equals(mSubtype.dataClass()))
        {
            output.format(
                "%sbuffer.get(arrayValue);%n", indent);
        }
        else
        {
            output.format(
                "%sfor (int i = 0; i < arraySize; ++i) {%n", indent);
            mSubtype.createDeserializer(
                field, "arrayValue[i]", indent1, output, false);
            output.format("%s}%n", indent);
        }

        // Now place the array into the builder.
        output.format("%sbuilder.%s(arrayValue);%n",
                      indent,
                      fieldName);

        return;
    } // end of createDeserializer(...)

    //
    // end of DataType Abstract Method Overrides.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get methods
    //

    /**
     * Returns the Java class name with "[]" appended.
     * @return Java class name.
     */
    @Override
    public String dataClassName()
    {
        return (String.format("%s[]", mSubtype.dataClassName()));
    } // end of dataClassName()

    /**
     * Returns the array subtype. All array items are of this
     * type.
     * @return the array subtype.
     */
    public DataType subtype()
    {
        return (mSubtype);
    } // end of subtype

    //
    // end of Get methods
    //-----------------------------------------------------------

    /**
     * Returns the subtype Java class name followed by "[]".
     * @return the subtype Java class name followed by "[]".
     */
    @Override
    public String toString()
    {
        return (String.format("%s[]", mSubtype));
    } // end of toString()
} // end of class ArrayType
