//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright (C) 2019. Charles W. Rapp.
// All Rights Reserved.
//

package net.sf.eBus.messages.type;

import java.nio.BufferOverflowException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Formatter;
import java.util.List;
import net.sf.eBus.messages.EMessage;
import net.sf.eBus.messages.EMessageObject;
import net.sf.eBus.messages.EReplyMessage;
import net.sf.eBus.messages.UnknownMessageException;
import net.sf.eBus.messages.ValidationException;


/**
 * This non-abstract class is used for
 * {@code abstract EMessageObject} classes. This message type
 * allows an {@code EMessage} and {@code EField} types to
 * contains abstract message fields. This allows a single message
 * class to be used for a range of field types. The downside is
 * that message serialization and de-serialization is slowed due
 * to the need to place the concrete class name in the serialized
 * message.
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public final class AbstractMessageType
    extends MessageType
{
//---------------------------------------------------------------
// Member data.
//

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new instance of AbstractMessageType.
     */
    /* package */ AbstractMessageType(final Class<?> clazz,
                                      final List<MessageField> fields,
                                      final List<Class<? extends EReplyMessage>> replies)
    {
        super (clazz, fields, replies);
    } // end of AbstractMessageType(...)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // DataType Abstract Method Overrides.
    //

    /**
     * Serializes the message fields to the given the buffer,
     * setting the field mask.
     * <p>
     * <strong>Note:</strong> as of eBus 4.4.0, messages are
     * automatically compiled when first used for serialization
     * or de-serialization. This increases the time it takes to
     * send or receive a message the first time. When a message
     * is frequently transmitted, it is recommended that the
     * message class is "compiled" by calling
     * {@link DataType#findType(java.lang.Class)} for that
     * message class. So when a message is sent or received the
     * first time, the message compilation is already done.
     * </p>
     * @param o serialize this message object.
     * @param buffer write to this buffer.
     * @throws BufferOverflowException
     * if {@code buffer} does not contain sufficient space to
     * store the message.
     */
    @Override
    public void serialize(final Object o,
                          final ByteBuffer buffer)
        throws BufferOverflowException
    {
        final Class<?> mc = this.getClass();
        final MessageType mt =
            (MessageType) DataType.findType(mc);

        // Serialize the concrete leaf class.
        CLASS_TYPE.serialize(mc, buffer);

        // If the concrete leaf class is a message, then
        // serialize the message subject.
        if (mt.isMessage())
        {
            STRING_TYPE.serialize(((EMessage) o).subject, buffer);
        }

        mt.serialize(o, buffer);

        return;
    } // end of serialize(Object, ByteBuffer)

    /**
     * Returns a message object de-serialized from the given
     * buffer.
     * <p>
     * <strong>Note:</strong> as of eBus 4.4.0, messages are
     * automatically compiled when first used for serialization
     * or de-serialization. This increases the time it takes to
     * send or receive a message the first time. When a message
     * is frequently transmitted, it is recommended that the
     * message class is "compiled" by calling
     * {@link DataType#findType(java.lang.Class)} for that
     * message class. So when a message is sent or received the
     * first time, the message compilation is already done.
     * </p>
     * @param buffer read in the message from this buffer.
     * @return the de-serialized message.
     * @throws IllegalStateException
     * if {@link #subject(String)} was not called prior to
     * de-serializing.
     * @throws BufferUnderflowException
     * if {@code buffer} contains an incomplete message.
     * @throws UnknownMessageException
     * if the serialized message is unknown to this eBus.
     * @throws ValidationException
     * if the message is incorrectly serialized.
     */
    @Override
    @SuppressWarnings ("unchecked")
    public Object deserialize(ByteBuffer buffer)
        throws IllegalStateException,
               BufferUnderflowException,
               UnknownMessageException,
               ValidationException
    {
        final Class<?> mc = (Class<?>) CLASS_TYPE.deserialize(buffer);
        final MessageType mt =
            (MessageType) DataType.findType(mc);

        // Get the subject if this is a message.
        if (mt.isMessage())
        {
            mt.subject((String) STRING_TYPE.deserialize(buffer));
        }

        // Now de-serialize the concrete message type.
        return (mt.deserialize(buffer));
    } // end of deserialize(ByteBuffer)

    /**
     * Adds the Java code used to serialize the named field to
     * a {@link ByteBuffer}.
     * @param field message field.
     * @param fieldName the message field name.
     * @param indent indent the code by this amount.
     * @param output write the code to this formatter.
     */
    @Override
    protected void createSerializer(final MessageField field,
                                    final String fieldName,
                                    final String indent,
                                    final Formatter output)
    {
        // 1. Get the field's class and message type.
        output.format("%sjava.lang.Class mc = (%s).getClass();%n",
                      indent,
                      fieldName);
        output.format("%snet.sf.eBus.messages.type.MessageType mt = (net.sf.eBus.messages.type.MessageType) net.sf.eBus.messages.type.DataType.findType(mc);%n%n",
                      indent);

        // 2. Serialize the field class to buffer. This is
        //    unavoidable since the class is needed to
        //    de-serialize the rest of the message. The result is
        //    that the message is no longer correctly byte
        //    aligned.
        output.format("%sCLASS_TYPE.serialize(mc, buffer);%n",
                      indent);

        // 3. If the field class is an EMessage subclass, then
        //    also serialize the message subject.
        output.format(
            "%sif (mt.isMessage()) {%n",
            indent);
        output.format(
            "%s  STRING_TYPE.serialize(((net.sf.eBus.messages.EMessage) %s).subject, buffer);%n",
            indent,
            fieldName);
        output.format("%s}%n%n", indent);

        // 4. Now serialize the message object.
        output.format(
            "%smt.serialize(%s, buffer);%n",
            indent,
            fieldName);

        return;
    } // end of createSerializer(...)

    /**
     * Generates the code to de-serialize the named field
     * from a {@link ByteBuffer}.
     * @param field message field.
     * @param fieldName store the de-serialized field in this
     * name.
     * @param indent indent the code by this amount.
     * @param output append the code to this formatter.
     * @param useBuilder if {@code true} then {@code fieldName}
     * is a builder method name; otherwise a local variable.
     */
    @Override
    protected void createDeserializer(final MessageField field,
                                      final String fieldName,
                                      final String indent,
                                      final Formatter output,
                                      final boolean useBuilder)
    {
        // 1. Deserialize the message object class and get its
        //    eBus data type.
        output.format(
            "%sjava.lang.Class mc = (java.lang.Class) CLASS_TYPE.deserialize(buffer);%n",
            indent);
        output.format(
            "%snet.sf.eBus.messages.type.MessageType mt = (net.sf.eBus.messages.type.MessageType) net.sf.eBus.messages.type.DataType.findType(mc);%n%n",
            indent);

        // 2. If the serialized message object is an EMessage
        //    subclass, then extract the message subject and put
        //    that subject into the message type.
        output.format(
            "%sif (mt.isMessage()) {%n",
            indent);
        output.format(
            "%s  mt.subject((java.lang.String) STRING_TYPE.deserialize(buffer));%n",
            indent);
        output.format("%s}%n%n", indent);

        // 3. Now de-serialize the message field mask and fields.
        final String format =
            (useBuilder ?
             "%sbuilder.%s((%s) mt.deserialize(buffer));%n" :
             "%s%s = (%s) mt.deserialize(buffer);%n");

        output.format(
            format, indent, fieldName, mClass.getName());

        return;
    } // end of createDeserializer(...)

    //
    // end of DataType Abstract Method Overrides.
    //-----------------------------------------------------------

    /**
     * Returns an {@link AbstractMessageType} for the given
     * abstract class.
     * @param jClass create message type for this class.
     * @return abstract message type.
     */
    /* package */ static DataType createAbstractMessageType(final Class<? extends EMessageObject> jClass)
    {
        final List<MessageField> fields = findFields(jClass);
        final List<Class<? extends EReplyMessage>> replyClasses =
            new ArrayList<>();

        if (EReplyMessage.class.isAssignableFrom(jClass))
        {
            replyClasses(jClass, replyClasses);
        }

        return (
            new AbstractMessageType(
                jClass, fields, replyClasses));
    } // end of createAbstractMessageType(Class)
} // end of class AbstractMessageType
