//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright (C) 2019. Charles W. Rapp.
// All Rights Reserved.
//

package net.sf.eBus.messages;

import com.typesafe.config.ConfigException.ValidationProblem;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * This exception is thrown when message build fails due to an
 * incorrectly configured message. This will be due to either
 * required fields not being set or two or more fields set to
 * incompatible values. That is setting one field to a given
 * value restricts the values allowed for one or more other
 * fields. This method is "fail slow" meaning that it should
 * contain all the {@link ValidationProblem}s found with the
 * configured message and <em>not</em> the first problem.
 * <p>
 * This exception should not be thrown if a single field is set
 * to an invalid value. Instead an exception is thrown by the
 * setter method when given that invalid value.
 * </p>
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public final class ValidationException
    extends RuntimeException
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * Serialization version identifier.
     */
    private static final long serialVersionUID = 0x050200L;

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * Build failed attempting to build this message object
     * class.
     */
    private final Class<? extends EMessageObject> mClass;

    /**
     * Text explaining each validation failure.
     */
    private final List<String> mProblems;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new instance of {@code ValidationException}
     * for the given message class and validation problems.
     * @param mc target message object class.
     * @param problems discovered message class problems.
     */
    public ValidationException(final Class<? extends EMessageObject> mc,
                               final List<String> problems)
    {
        super (asMessage(mc, problems));

        mClass = mc;
        mProblems = new ArrayList<>(problems);
    } // end of ValidationException(Class, List)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Object Method Overrides.
    //

    /**
     * Returns text displaying the message validation problems.
     * @return exception in text format.
     */
    @Override
    public String toString()
    {
        final StringBuilder retval = new StringBuilder();

        retval.append(mClass.getCanonicalName())
              .append(" validation failures:")
              .append(getMessage());

        return (retval.toString());
    } // end of toString()

    //
    // end of Object Method Overrides.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get methods.
    //

    /**
     * Returns the message class experiencing the validation
     * problems.
     * @return message object class.
     */
    public Class<? extends EMessageObject> messageClass()
    {
        return (mClass);
    } // end of messageClass()

    /**
     * Returns the validation problems as a read-only list.
     * @return problems list.
     */
    public Iterable<String> problems()
    {
        return (Collections.unmodifiableList(mProblems));
    } // end of problems()

    //
    // end of Get methods.
    //-----------------------------------------------------------

    /**
     * Returns text displaying the message validation problems.
     * @param mc invalid message class.
     * @param problems invalid issues list.
     * @return exception in text format.
     */
    private static String asMessage(final Class<? extends EMessageObject> mc,
                                    final List<String> problems)
    {
        final StringBuilder retval = new StringBuilder();

        retval.append(mc.getCanonicalName())
              .append(" failed to build due to the following problems:");

        problems.forEach(
            problem -> retval.append('\n').append(problem));

        return (retval.toString());
    } // end of asMessage(Class, List<>)
} // end of ValidationException
