//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright (C) 2001 - 2008, 2015. Charles W. Rapp.
// All Rights Reserved.
//

package net.sf.eBus.messages;

/**
 * This exception is thrown when an unknown message identifier
 * is referenced. This will only occur when deserializing an
 * incoming message.
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public final class UnknownMessageException
    extends RuntimeException
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * This is eBus version 2.1.0.
     */
    private static final long serialVersionUID = 0x050200L;

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * The message class name causing this exception.
     */
    private final String mClass;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Constructs an exception for the given message identifier.
     * @param clazz the message class name.
     */
    public UnknownMessageException(final String clazz)
    {
        super (String.format("%s unknown", clazz));

        mClass = clazz;
    } // end of UnknownMessageException(String)

    /**
     * Constructs an exception for the given message identifier
     * and the exception explaining when this message is unknown.
     * @param clazz the message class name.
     * @param cause the exception which caused this exception.
     */
    public UnknownMessageException(final String clazz,
                                   final Throwable cause)
    {
        super (String.format("%s unknown", clazz), cause);

        mClass = clazz;
    } // end of UnknownMessageException(String, Throwable)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get methods.
    //

    /**
     * Returns the message class name.
     * @return the message class name.
     */
    public String messageClass()
    {
        return (mClass);
    } // end of messageClass()

    //
    // end of Get methods.
    //-----------------------------------------------------------
} // end of class UnknownMessageException

//
// CHANGE LOG
// $Log: UnknownMessageException.java,v $
// Revision 1.3  2007/11/12 15:31:38  charlesr
// Updated Version import.
//
// Revision 1.2  2005/12/31 12:59:51  charlesr
// Changed getName() and getVersion() to name() and
// version(), respectively.
//
// Revision 1.1  2005/03/11 19:47:07  charlesr
// Replaced MessageName class with String.
//
// Revision 1.0  2004/08/04 14:51:01  charlesr
// Initial revision
//
