//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright (C) 2001 - 2008, 2013. Charles W. Rapp.
// All Rights Reserved.
//

package net.sf.eBus.messages;

/**
 * {@code InvalidMessageException} is thrown when a message
 * class has an invalid layout. A message class is invalid when:
 * <ul>
 *   <li>
 *     if the class is not an {@link EMessage} subclass,
 *   </li>
 *   <li>
 *     a message field is either not public, not final or is
 *     static, or
 *   </li>
 *   <li>
 *     the message class does not have the required deserialize
 *     builder inner class (see {@link EMessageObject} for more
 *     about this).
 *   </li>
 * </ul>
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public final class InvalidMessageException
    extends RuntimeException
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * This is eBus version 2.1.0.
     */
    private static final long serialVersionUID = 0x050200L;

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * The invalid message identifier.
     */
    private final Class<? extends EMessageObject> mMsgClass;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates an InvalidMessageException for the given message.
     * @param mc the invalid message class.
     */
    public InvalidMessageException(
        final Class<? extends EMessageObject> mc)
    {
        super ("invalid message");

        mMsgClass = mc;
    } // end of InvalidMessageException(Class)

    /**
     * Creates an InvalidMessageException for the given message
     * and exception reason.
     * @param mc the invalid message class.
     * @param reason the reason the message is invalid.
     */
    public InvalidMessageException(
        final Class<? extends EMessageObject> mc,
        final String reason)
    {
        super (reason);

        mMsgClass = mc;
    } // end of InvalidMessageException(Class, String)

    /**
     * Creates an InvalidMessageException for the given message,
     * reason, and cause.
     * @param mc the invalid message class.
     * @param reason the reason the message is invalid.
     * @param t the cause for this exception.
     */
    public InvalidMessageException(
        final Class<? extends EMessageObject> mc,
        final String reason,
        final Throwable t)
    {
        super (reason, t);

        mMsgClass = mc;
    } // end of InvalidMessageException(Class, String, Throwable)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get methods.
    //

    /**
     * Returns the message class.
     * @return the message class.
     */
    public Class<? extends EMessageObject> messageClass()
    {
        return (mMsgClass);
    } // end of messageClass()

    //
    // end of Get methods.
    //-----------------------------------------------------------
} // end of class InvalidMessageException
