//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2013, 2016, 2019. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.messages;

import java.io.Serializable;

/**
 * Required base class for all system messages. Note that system
 * messages may <em>not</em> be used by applications. If an eBus
 * application extends this class, it will not be able to send
 * or received such a message. System messages are meant for
 * internal eBus use only.
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public abstract class ESystemMessage
    extends EMessage
    implements Serializable
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * All system messages use the same subject: "/eBus".
     */
    public static final String SYSTEM_SUBJECT = "/eBus";

    /**
     * Serialization version identifier.
     */
    private static final long serialVersionUID = 0x050200L;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates an eBus system message using the default values:
     * {@link #SYSTEM_SUBJECT /eBus subject}, and the current
     * time.
     *
     * @see #ESystemMessage(String, long)
     *
     * @deprecated use
     * {@link #ESystemMessage(ESystemMessage.Builder)} in the
     * future.
     */
    @Deprecated
    protected ESystemMessage()
        throws IllegalArgumentException
    {
        super (SYSTEM_SUBJECT,
               System.currentTimeMillis(),
               MessageType.SYSTEM);
    } // end of ESystemMessage(String)

    /**
     * Creates a new system message instance with the given
     * subject and timestamp.
     * @param subject the required message subject.
     * @param timestamp the message timestamp in Java millisecond
     * epoch time.
     * @throws IllegalArgumentException
     * if {@code subject} is either of {@code null} or empty.
     *
     * @deprecated use
     * {@link #ESystemMessage(ESystemMessage.Builder)} in the
     * future.
     */
    @Deprecated
    protected ESystemMessage(final String subject,
                             final long timestamp)
        throws IllegalArgumentException
    {
        super (subject, timestamp, MessageType.SYSTEM);
    } // end of ESystemMessage(String, long)

    /**
     * Creates a new system message based on the given message
     * builder. {@code builder} is guaranteed to contain a valid
     * message configuration at this point.
     * @param builder contains the eBus message configuration.
     */
    protected ESystemMessage(final Builder<?, ?> builder)
    {
        super (builder);
    } // end of ESystemMessage(EMessageBuilder)

    //
    // end of Constructors.
    //-----------------------------------------------------------

//---------------------------------------------------------------
// Inner classes.
//

    /**
     * Base class for all {@link ESystemMessage} builders. Used
     * by eBus when de-serializing an encoded message back into
     * the target message object.
     *
     * @param <M> target message class.
     * @param <B> concrete (leaf) builder subclass.
     */
    @SuppressWarnings ("unchecked")
    public static abstract class Builder<M extends ESystemMessage,
                                         B extends Builder<M, ?>>
        extends EMessage.Builder<M, B>
    {
    //-----------------------------------------------------------
    // Member data.
    //

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        protected Builder(final Class<? extends EMessageObject> targetClass)
        {
            super (targetClass,
                   SYSTEM_SUBJECT,
                   System.currentTimeMillis(),
                   MessageType.SYSTEM);
        } // end of Builder(Class)

        //
        // end of Constructors.
        //-------------------------------------------------------
    } // end of class Builder
} // end of class ESystemMessage
