//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2013, 2015, 2016. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.messages;

import java.io.Serializable;

/**
 * Required base class for all application request messages.
 * This class has no fields of its own. This class allows
 * {@link EReplyMessage} to be used to reply to any and all
 * request message classes.
 * <p>
 * {@code ERequestMessage} subclasses must contain a class-level
 * annotation {@link EReplyInfo}. This defines the reply message
 * classes used to reply to the request message. This annotation
 * is cumulative. The allowed reply messages include those listed
 * in the current {@code EReplyInfo} annotation and those listed
 * in the superclass {@code EReplyInfo} annotation. Since
 * {@code EReplyMessage} is listed in the {@code ERequestMessage}
 * {@code EReplyInfo} annotation, {@code EReplyMessage} is a
 * valid reply to any request.
 * </p>
 * <pre>
 *   <code>
 * {@literal @}EReplyInfo (replyMessageClasses = {OrderReply.class})
 * public final class OrderRequest
 *     extends ERequestMessage
 *   </code>
 * </pre>
 *
 * @see ENotificationMessage
 * @see EReplyMessage
 * @see EReplyInfo
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

@EReplyInfo (replyMessageClasses = {EReplyMessage.class})
public abstract class ERequestMessage
    extends EMessage
    implements Serializable
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * Serialization version identifier.
     */
    private static final long serialVersionUID = 0x050200L;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates an eBus request message with the given subject.
     * The message timestamp is set to the current time.
     * @param subject the request message subject.
     * @throws IllegalArgumentException
     * if {@code subject} is either {@code null} or empty.
     */
    protected ERequestMessage(final String subject)
        throws IllegalArgumentException
    {
        super (subject,
               System.currentTimeMillis(),
               MessageType.REQUEST);
    } // end of ERequestMessage(String)

    /**
     * Creates a new request message instance for the given
     * subject and timestamp. This constructor is used for
     * message de-serialization.
     * <p>
     * This constructor is <em>not</em> deprecated since it is
     * needed by local messages designated by
     * {@code &commat;ELocalOnly} annotation.
     * </p>
     * @param subject the message subject.
     * @param timestamp the message timestamp (Java millisecond
     * epoch time).
     * @throws IllegalArgumentException
     * if {@code subject} is either of {@code null} or empty.
     */
    public ERequestMessage(final String subject,
                           final long timestamp)
        throws IllegalArgumentException
    {
        super (subject,
               timestamp,
               MessageType.REQUEST);
    } // end of ERequestMessage(String, long)

    /**
     * Creates a new eBus request message based on the given
     * message builder. {@code builder} is guaranteed to contain
     * a valid message configuration at this point.
     * @param builder contains the eBus message configuration.
     */
    protected ERequestMessage(final Builder<?, ?> builder)
    {
        super (builder);
    } // end of ERequestMessage(EMessageBuilder)

    //
    // end of Constructors.
    //-----------------------------------------------------------

//---------------------------------------------------------------
// Inner classes.
//

    @SuppressWarnings ("unchecked")
    public static abstract class Builder<M extends ERequestMessage,
                                         B extends Builder<M, ?>>
        extends EMessage.Builder<M, B>
    {
    //-----------------------------------------------------------
    // Member data.
    //

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        protected Builder(final Class<? extends EMessageObject> targetClass)
        {
            super (targetClass, MessageType.REQUEST);
        } // end of Builder(Class)

        //
        // end of Constructors.
        //-------------------------------------------------------
    } // end of class Builder
} // end of class ERequestMessage
