//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2013, 2016, 2019. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.messages;

import java.io.Serializable;

/**
 * Required base class for all application publish/subscribe
 * notification messages. This class has no fields of its own.
 *
 * @see ERequestMessage
 * @see EReplyMessage
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public abstract class ENotificationMessage
    extends EMessage
    implements Serializable
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * Serialization version identifier.
     */
    private static final long serialVersionUID = 0x050200L;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates an eBus notification message with the given
     * subject. The message timestamp is set to the current time.
     * <p>
     * This constructor is <em>not</em> deprecated since it is
     * needed by local messages designated by
     * {@code &commat;ELocalOnly} annotation.
     * </p>
     * @param subject the notification subject.
     * @throws IllegalArgumentException
     * if {@code subject} is either {@code null} or empty.
     */
    protected ENotificationMessage(final String subject)
        throws IllegalArgumentException
    {
        super (subject,
               System.currentTimeMillis(),
               MessageType.NOTIFICATION);
    } // end of ENotificationMessage(String)

    /**
     * Creates a new notification message instance for the given
     * subject and timestamp. This constructor is used for
     * message de-serialization.
     * <p>
     * This constructor is {@code public} due to automatic
     * message compilation and generic message field
     * requirements.
     * </p>
     * <p>
     * This constructor is <em>not</em> deprecated since it is
     * needed by local messages designated by
     * {@code &commat;ELocalOnly} annotation.
     * </p>
     * @param subject the message subject.
     * @param timestamp the message timestamp (Java millisecond
     * epoch time).
     * @throws IllegalArgumentException
     * if {@code subject} is either of {@code null} or empty.
     */
    public ENotificationMessage(final String subject,
                                final long timestamp)
        throws IllegalArgumentException
    {
        super (subject,
               timestamp,
               MessageType.NOTIFICATION);
    } // end of ENotificationMessage(String, long)

    /**
     * Creates a new eBus notification message based on the given
     * message builder. {@code builder} is guaranteed to contain
     * a valid message configuration at this point.
     * @param builder contains the eBus notification message
     * configuration.
     */
    protected ENotificationMessage(final Builder<?, ?> builder)
    {
        super (builder);
    } // end of ENotificationMessage(EMessageBuilder)

    //
    // end of Constructors.
    //-----------------------------------------------------------

//---------------------------------------------------------------
// Inner classes.
//

    /**
     * Base class for all {@link ENotificationMessage} builders.
     * Used by eBus when de-serializing an encoded message back
     * into the target message object.
     *
     * @param <M> builds this target message class.
     * @param <B> message builder subclass. Needed to return the
     * correct builder type when setting fields. If this were not
     * the case, field chaining would not work.
     */
    @SuppressWarnings ("unchecked")
    public static abstract class Builder<M extends ENotificationMessage,
                                         B extends Builder<M, ?>>
        extends EMessage.Builder<M, B>
    {
    //-----------------------------------------------------------
    // Member data.
    //

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * Creates a new notification message builder for the
         * given message class.
         * @param targetClass notification message sub-class.
         */
        protected Builder(final Class<? extends EMessageObject> targetClass)
        {
            super (targetClass, MessageType.NOTIFICATION);
        } // end of Builder(Class)

        /**
         * Creates a new notification message builder for the
         * given message class and subject.
         * @param targetClass notification message sub-class.
         * @param subject message subject.
         */
        protected Builder(final Class<? extends EMessageObject> targetClass,
                          final String subject)
        {
            super (targetClass, subject, MessageType.NOTIFICATION);
        } // end of Builder(Class, String)

        //
        // end of Constructors.
        //-------------------------------------------------------
    } // end of class Builder
} // end of class ENotificationMessage
