//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2013, 2016, 2019. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.messages;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import net.sf.eBus.messages.type.DataType;

/**
 * This abstract class defines the two eBus message classes:
 * {@link EMessage} and {@link EField}. Message instances are
 * used to publish notifications, requests, and replies. Field
 * instances are used to implement user-defined fields within
 * both messages and other fields.
 * <p>
 * Messages and field contain {@code public final} data members.
 * If the message or field can be transmitted between eBus
 * applications, then the data member type must be one of the
 * following:
 * </p>
 * <pre><code>boolean Boolean BigInteger BigDecimal byte Byte char Character
 Class Date double Double Duration EField EFieldList EMessage
 EMessageList EMessageKey enum File float Float InetAddress
 InetSocketAddress Instant int Integer LocalDate LocalDateTime
 LocalTime long Long MonthDay OffsetDateTime OffsetTime Period
 short Short String URI YearMonth ZonedDateTime ZoneId ZoneOffset</code></pre>
 * <p>
 * Any eBus field can be made into a homogenous array by
 * appending {@code []} to the field name.
 * </p>
 * <p>
 * This data type restriction does <em>not</em> apply if the
 * message or field is marked as {@link ELocalOnly local only}.
 * In that case the data members may be any valid Java type.
 * Also the following De-serialization requirements do not apply.
 * However, local only message and fields may not appear in a
 * non-local message or field.
 * </p>
 * <h1>De-serialization</h1>
 * <p>
 * As of eBus release 5.2.0, eBus determines field member
 * serialization order based on the field's length. The goal
 * being to maintain correct by alignment. Because serialization
 * order can no longer be determined up front, {@code EMessage}
 * and {@code EField} instances must be re-created using a
 * technique not based on field ordering. This is no done using
 * the GoF builder paradigm.
 * </p>
 * <p>
 * Consider the following eBus notification message:
 * </p>
 * <pre><code>public final class EquityQuote
    extends ENotificationMessage
    implements Serializable
{
    <em>// Member fields.</em>
    public final PriceType priceType; <em>// PriceType is an enum</em>
    public final PriceSize bid;       <em>// PriceSize is an EField subclass</em>
    public final PriceSize ask;

    private static final long serialVersionUID = 1L; <em>// Defined for Serializable</em>

    <em>// Remaining code will be added by each step.</em>
}</code></pre>
 * <h2>Step 1: builder method</h2>
 * <p>
 * Every non-local {@code EMessage} and {@code EField} must
 * define a {@code public static builder} method which returns a
 * builder instance associated with the message or field.
 * The builder class name is left up to the developer. In this
 * example the class is simply {@code Builder}:
 * </p>
 * <pre><code>public static Builder builder() {
    return (new Builder());
}</code></pre>
 * <p>
 * It is recommended that this {@code builder} method be the only
 * way to create a {@code EquityQuote.Builder} instance.
 * </p>
 * <h2>Step 2: Declare builder class</h2>
 * <pre><code>public static final class Builder
    extends ENotificationMessage.Builder&lt;EquityQuote, Builder&gt;
{
    <em>// This code filled in by the following steps.</em>
}</code></pre>
 * <p>
 * The builder class <em>must</em> extend the target message's
 * base class builder. In this example, since {@code EquityQuote}
 * extends {@code ENotificationMessage}, {@code Builder} must
 * extend {@link ENotificationMessage.Builder}. If the message
 * extends request or reply, then the builder would be required
 * to extend the request or reply builder, respectively.
 * </p>
 * <p>
 * {@code ENotification.Builder} generic class parameters
 * {@code M} and {@code B} (which all super class Builder classes
 * define) are the target message class and the target builder
 * class, respectively. The first is needed for the superclass
 * {@code public final M build()} method used to create the
 * target message instance. The second generic parameter is used
 * by super class setter methods (set Step 5 for more about
 * setters).
 * </p>
 * <p>
 * The builder inner class must be {@code public static} because
 * classes outside this package will need to access the class and
 * {@code static} because the builder instance must in no way be
 * dependent on the target message class containing it.
 * </p>
 * <h2>Step 3: Builder fields</h2>
 <pre><code>private PriceType mPriceType;
private PriceSize bid;
private PriceSize ask;</code></pre>
 * <p>
 * The builder class must have exactly the same number of fields
 * as the target message with the same data types <em>except</em>
 * this time these fields are {@code private}, non-{@code final}.
 * {@code private} because only the encapsulating target class is
 * allowed to access these fields. Non-{@code final} because the
 * field values will be set post-construction.
 * </p>
 * <h2>Step 4: Builder constructor</h2>
 * <p>
 * A builder should have only one {@code private}, no argument
 * constructor because only {@code EquityQuote builder} method
 * should be allowed to construct this builder:
 * </p>
 * <pre><code> private Builder() {
    <em>// Superclass constructor requires target class.</em>
    super (EquityQuote.class);

    <em>// Set fields to defaults if necessary.</em>
}</code></pre>
 * <h2>Step 5: Setter methods</h2>
 * <p>
 * For each {@code public final} target message field the builder
 * <em>must</em> define a setter method whose name
 * <em>exactly</em> matches the field name and whose sole
 * parameter has <em>exactly</em> matches the field data type.
 * The setter return type is the builder class which allows for
 * setter method chaining:
 * {@code builder.height(1).weight(2).age(3)}. This is the reason
 * for the second class parameter {@code B}. If a message field
 * is defined in the superclass, then the superclass builder is
 * responsible for defining the setter method. But if the
 * superclass setter returns the superclass type, then setter
 * chain breaks. The solution is for superclass setters to return
 * type {@code B}. While this requires a downcast it will work as
 * long as {@code B} was set correctly.
 * </p>
 * <p>
 * In this example {@code EquityQuote.Builder} has the following
 * setters:
 * </p>
 * <pre><code>public Builder priceType(final PriceType pt) { ... };
public Builder bid(final PriceSize ps) { ... };
public Builder ask(final PriceSize ps) { ... };</code></pre>
 * <p>
 * How the setter is defined depends on what argument validation
 * is required. If no validation is done, then the setter can
 * set the field value and return. If validation is done, then
 * an invalid argument should result in a thrown exception.
 * </p>
 * <pre><code>public Builder bid(final PriceSize ps) {
    if (ps == null) {
        throw (new NullPointerException("ps is null"));
    }

    mBid = ps;
    return (this);
}</code></pre>
 * <p>
 * If validation is performed then that validation must be
 * independent of the other data members and focus on the setter
 * argument but itself. This is due to the unknown ordering of
 * the setter method calls. When a setter is called, there is
 * no way of knowing which builder fields are set, if any. If
 * inter-field validation is required, then see Step 6 to learn
 * how a builder can implement this requirement.
 * </p>
 * <h2>Step 6: Builder validation</h2>
 * <p>
 * Before the target {@code EMessage} or {@code EField} is
 * instantiated, builder member fields may be checked for
 * correctness. Application code has finished calling the setters
 * and called {@link EMessageObject.Builder#build}. This method
 * then calls {@link EMessageObject.Builder#validate(List)} to
 * determine if the builder is correctly configured. The
 * {@code validate} method should <em>not</em> throw an exception
 * but rather adds text to the {@code problems} list explaining
 * the problems found with the builder configuration. Upon return
 * an empty list means the builder is properly configured and the
 * target message can now be built. A non-empty problems list
 * results in a {@link ValidationException} being thrown which
 * contains the problems list.
 * </p>
 * <p>
 * When overridding {@code validate} use a series of {@code if}
 * statements for each check rather than {@code if/else if}
 * chain. The point here is that all builder configuration
 * problems should be reported rather than just the first one.
 * This way the developer will be able to correct all the
 * problems at one time rather than uncovering the errors one
 * at a time. Finally, the first line in {@code validate} should
 * be {@code super.validate(problems);} to allow the superclass
 * to perform its own validation.
 * </p>
 * <pre><code>&#64;Override public void validate(final List&lt;String&gt; problems) {
    super.validate(problems);

    if (mPriceType == null) { problems.add("priceType not set"); }

    <em>// One-sided bids are allowed.</em>
    <em>// Note: inter-field comparison may now be performed.</em>
    if (mBid == null &amp;&amp; mAsk == null) { problems.add("bid and ask both not set"); }

    <em>// Bid price must be less than ask price. PriceSize implements Comparable.</em>
    <em>// Be sure this is a two-sided quote.</em>
    if (mBid != null &amp;&amp; mAsk != null &amp;&amp; mBid.compareTo(mAsk) &ge; 0) {
        problems.add("bid &ge; ask");
    }
}</code></pre>
 * <h2>Step 7: Build target</h2>
 * <p>
 * The abstract method the builder class is required to override
 * is {@code buildImpl}. This class returns the target class
 * instance built from the configured fields.
 * </p>
 * <pre><code>&#64;Override protected EquityQuote buildImpl() {
    return (new EquityQuote(this));
}</code></pre>
 * <p>
 * {@code buildImpl} is called after {@code validate} shows no
 * problems with the builder configuration.
 * </p>
 * <h2>Step 8: Target class constructor</h2>
 * <p>
 * The final link connecting the target class and target class
 * builder is the target class constructor:
 * </p>
 * <pre><code>private EquityQuote(final Builder builder) {
    super (builder);

    this.priceType = builder.mPriceType;
    this.bid = builder.mBid;
    this.ask = builder.mAsk;
}</code></pre>
 * <p>
 * This constructor is {@code private} because only
 * {@code EquityQuote.Builder.buildImpl} has access to this
 * constructor. Also note that since {@code Builder} is an
 * {@code EquityQuote} inner class, the constructor can access
 * {@code Builder} member fields directly, no getter methods are
 * required.
 * </p>
 * <h2>Step 9: Complete message class definition</h2>
 * <pre><code>public final class EquityQuote
     extends ENotificationMessage
     implements Serializable
{
    public final PriceType priceType;
    public final PriceSize bid;
    public final PriceSize ask;

    private static final long serialVersionUID = 1L;

    private EquityQuote(final Builder builder) {
        super (builder);

        this.priceType = builder.mPriceType;
        this.bid = builder.mBid;
        this.ask = builder.mAsk;
    }

    public static Builder builder() {
        return (new Builder());
    }

    public static final class Builder
        extends ENotificationMessage.Builder&lt;EquityQuote, Builder&gt;
    {
        private PriceType mPriceType;
        private PriceSize mBid;
        private PriceSize mAsk;

        private Builder() {
            super (EquityQuote.class);
        }

        public Builder priceType(final PriceType pt) {
            if (pt == null) {
                throw (new NullPointerException("pt is null"));
            }

            mPriceType = pt;
            return (this);
        }

        public Builder bid(final PriceSize ps) {
            if (ps == null) {
                throw (new NullPointerException("ps is null"));
            }

            mBid = ps;
            return (this);
        }

        public Builder ask(final PriceSize ps) {
            if (ps == null) {
                throw (new NullPointerException("ps is null"));
            }

            mAsk = ps;
            return (this);
        }

        &#64;Override public void validate(final List&lt;String&gt; problems) {
            super.validate(problems);

            if (mPriceType == null) { problems.add("priceType not set"); }
            if (mBid == null &amp;&amp; mAsk == null) { problems.add("bid and ask not set"); }
            if (mBid != null &amp;&amp; mAsk != null &amp;&amp; mBid.compareTo(mAsk) &ge; 0) {
                problems.add("bid &ge; ask");
            }
        }

        &#64;Override protected EquityQuote buildImpl() {
            return (new EquityQuote(this));
        }
    }
}</code></pre>
 * <p>
 * The above example uses a {@code final} message class. But what
 * if the {@code EMessage} or {@code EField} class may be
 * extended? See {@link EField} for an example of how to set up
 * builders for a non-{@code final} class.
 * </p>
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public abstract class EMessageObject
    implements Serializable
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * A message object may have at most 31 fields. This is
     * because there are 31 usable bits in a 4-byte, signed
     * integer.
     */
    public static final int MAX_FIELDS = 31;

    /**
     * Serialization version identifier.
     */
    private static final long serialVersionUID = 0x050200L;

    //-----------------------------------------------------------
    // Statics.
    //

    /**
     * Maps the {@code EMessageObject} class to its validation
     * flag: {@code null} means the class has not been validated,
     * {@code true} means the class is valid, and {@code false}
     * means invalid.
     */
    private static final ConcurrentMap<Class<? extends EMessageObject>, ValidationInfo> mValidMap =
        new ConcurrentHashMap<>();

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new message object instance. The point behind
     * this constructor is to validate a user-defined message
     * or field as soon as it is instantiated.
     * @throws InvalidMessageException
     * if the user-defined message or field is invalid.
     */
    protected EMessageObject()
        throws InvalidMessageException
    {
        final Class<? extends EMessageObject> mc =
            this.getClass();
        ValidationInfo info = mValidMap.get(mc);

        // Has this message been validated before?
        if (info == null)
        {
            // No, validate it now and store away the results.
            info = validate(mc);
            mValidMap.put(mc, info);
        }

        if (info.isValid() == false)
        {
            throw (
                new InvalidMessageException(
                    mc, info.reason(), info.error()));
        }
    } // end of EMessageObject()

    protected EMessageObject(final Builder<?, ?> builder)
    {}

    //
    // end of Constructors.
    //-----------------------------------------------------------

    /**
     * Returns the validation information for the specified
     * {@code EMessageObject} sub-class.
     * @param mc validate this message class.
     * @return the validation result.
     */
    private static ValidationInfo validate(final Class<? extends EMessageObject> mc)
    {
        boolean flag = true;
        String reason = null;
        Throwable t = null;

        try
        {
            // Called for effect only. This will create a
            // MessageType for this class. If mc is not a valid
            // message, then an InvalidMessageException is
            // thrown with the appropriate reason.
            DataType.findType(mc);
        }
        catch (InvalidMessageException msgex)
        {
            flag = false;
            reason = msgex.getMessage();
            t = msgex;
        }

        return (new ValidationInfo(flag, reason, t));
    } // end of validate(Class)

//---------------------------------------------------------------
// Inner classes.
//

    /**
     * Base class for all {@link EMessageObject} builders. Used
     * by eBus when de-serializing an encoded message back into
     * the target message object.
     *
     * @param <M> builds this target message class.
     * @param <B> message builder subclass. Needed to return the
     * correct builder type when setting fields. If this were not
     * the case, field chaining would not work.
     */
    @SuppressWarnings ("unchecked")
    public static abstract class Builder<M extends EMessageObject,
                                         B extends Builder<M, ?>>
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        /**
         * Message class instance returned by {@link #build}.
         * This value is used to identify the target class when
         * a build failure occurs, placed into the
         * {@link ValidationException}.
         */
        protected final Class<? extends EMessageObject> mTargetClass;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        protected Builder(final Class<? extends EMessageObject> targetClass)
        {
            mTargetClass = targetClass;
        } // end of Builder(Class)

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Abstract Method Declarations.
        //

        /**
         * Returns eBus message instance built from the
         * configured properties. <strong>Note:</strong> the
         * builder configuration was
         * {@link #validate(List) validated} prior to calling
         * this method. The message object may now be
         * constructed.
         * @return target message class instance.
         */
        protected abstract M buildImpl();

        /**
         * Checks if message subject and message type are
         * configured. If not, then text explaining this error
         * is appended to {@code problems}.
         * <p>
         * This method should be overridden by subclass message
         * builders and called <em>before</em> doing its own
         * validation. The first line in the subclass
         * {@code validate} implementation should be
         * {@code super.validate(problems);}.
         * </p>
         * <p>
         * When overriding this method, be sure to add all
         * discovered validation problems to the list. The
         * validation method should consist of a series of
         * individual {@code if} statements and <em>not</em>
         * an {@code if/else if} chain. That way all problems
         * are found and not just the first one.
         * </p>
         * @param problems append validation problems to this
         * list.
         */
        protected void validate(final List<String> problems)
        {}

        //
        // end of Abstract Method Declarations.
        //-------------------------------------------------------

        /**
         * Returns the target message instance constructed from
         * the configured properties. This must be the final
         * method in a builder configuration call chain.
         * @return target message instance.
         * @throws ValidationException
         * if {@code this} builder does not contain a valid
         * target message configuration. This exception contains
         * a list of all validation problems found.
         */
        public final M build()
            throws ValidationException
        {
            final List<String> problems = new ArrayList<>();

            validate(problems);
            if (!problems.isEmpty())
            {
                throw (
                    new ValidationException(
                        mTargetClass, problems));
            }

            return (buildImpl());
        } // end of build()
    } // end of class Builder

    /**
     * This immutable class tracks the EMessageObject validity.
     */
    private static final class ValidationInfo
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        /**
         * {@code true} if the message is valid and
         * {@code false} otherwise.
         */
        private final boolean mFlag;

        /**
         * Text explaining why {@code _flag} is {@code false}.
         * Set to {@code null} when {@code _flag} is
         * {@code true}.
         */
        private final String mReason;

        /**
         * Thrown error which caused the exception.
         */
        private final Throwable mError;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * Sets the message object validation to the given flag
         * and reason.
         * @param flag {@code true} if the message object is
         * valid and {@code false} otherwise.
         * @param reason text explaining the invalidity.
         * @param t error behind a {@code false flag}.
         */
        private ValidationInfo(final boolean flag,
                               final String reason,
                               final Throwable t)
        {
            mFlag = flag;
            mReason = reason;
            mError = t;
        } // end of ValidationInfo(boolean, String, Throwable)

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Get Methods.
        //

        /**
         * Returns {@code true} if the message object is valid
         * and {@code false} otherwise.
         * @return {@code true} if the message object is valid.
         */
        private boolean isValid()
        {
            return (mFlag);
        } // end of isValid()

        /**
         * Returns text explaining why the message is invalid.
         * May return {@code null}.
         * @return text explaining why the message is invalid.
         */
        private String reason()
        {
            return (mReason);
        } // end of reason()

        /**
         * Returns the error behind an invalid message.
         * @return {@code Throwable} error.
         */
        private Throwable error()
        {
            return (mError);
        } // end of error()

        //
        // end of Get Methods.
        //-------------------------------------------------------
    } // end of class ValidationInfo
} // end of class EMessageObject

