//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2017. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.messages;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;

/**
 * This {@link EField message field type} allows multiple
 * messages to be transported within another
 * {@link EMessage eBus message}. A message list is message key
 * homogenous. That is, a message list takes an eBus message key
 * parameter and uses that key to verify that all messages placed
 * in this list have the same key as the parameter. That also
 * means that {@code null} messages may not be added to a
 * message list.
 * <p>
 * The difference between {@code EMessageList} and
 * {@code EMessage[]} is that {@code EMessageList} serializes
 * the defining {@link EMessageKey} once and then each message
 * instance in turn. This works because {@code EMessageList}
 * items all use the same message key. {@code EMessage[]} allows
 * messages with different message keys, so the message key is
 * serialized for each message item. This greatly increases the
 * serialization/de-serialization time and serialized buffer
 * size.
 * </p>
 * <p>
 * This class extends {@link ArrayList}, providing all the
 * features available in that class.
 * </p>
 * <p>
 * There is no limit as to the number of elements which may be
 * placed into this list. But care must be taken not to overflow
 * the eBus {@code ByteBuffer} when serializing for transport.
 * </p>
 *
 * @param <E> specifies the homogenous, concrete {@code EMessage}
 * type stored in this list.
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public final class EMessageList<E extends EMessage>
    extends EAbstractList<E>
    implements Serializable,
               Cloneable
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * Java serializable unique identifier.
     */
    private static final long serialVersionUID = 0x050200L;

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * All messages placed into this list must have this key.
     */
    private final EMessageKey mKey;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates an empty message list instance with an initial
     * capacity of ten.
     * @param key list elements must be
     * {@link EMessage eBus messages} containing this message key.
     * @throws NullPointerException
     * if {@code key} is {@code null}.
     */
    public EMessageList(final EMessageKey key)
        throws NullPointerException
    {
        super ();

        if (key == null)
        {
            throw (new NullPointerException("key is null"));
        }

        mKey = key;
        mReadOnlyFlag = false;
    } // end of EMessageList(EMessageKey)

    /**
     * Creates an empty message list instance with the specified
     * initial capacity.
     * @param initialCapacity the list initial capacity.
     * @param key list elements must be
     * {@link EMessage eBus messages} containing this message key.
     * @throws NullPointerException
     * if {@code key} is {@code null}.
     */
    public EMessageList(final int initialCapacity,
                        final EMessageKey key)
        throws NullPointerException
    {
        super (initialCapacity);

        if (key == null)
        {
            throw (new NullPointerException("key is null"));
        }

        mKey = key;
        mReadOnlyFlag = false;
    } // end of EMessageList(int, EMessageKey)

    /**
     * Creates a message list instance containing the same
     * elements as {@code c} and in the order returned by the
     * collection's iterator.
     * @param c place this collection's elements into this
     * message list.
     * @param key list elements must be
     * {@link EMessage eBus messages} containing this message key.
     * @throws NullPointerException
     * if either {@code c} or {@code key} is {@code null}.
     * @throws IllegalArgumentException
     * if {@code c} contains a message with a key that does not
     * equal {@code key}.
     */
    public EMessageList(final Collection<E> c,
                        final EMessageKey key)
        throws NullPointerException,
               IllegalArgumentException
    {
        super (c);

        if (key == null)
        {
            throw (new NullPointerException("key is null"));
        }

        validate(c, key);

        mKey = key;
        mReadOnlyFlag = false;
    } // end of EMessageList(Collection<>, EMessageKey)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // EAbstractList Method Overrides.
    //

    /**
     * Appends the specified element to the end of this list.
     * @param e append this element to list.
     * @return {@code true} (as specified by
     * {@code Collection.add(E)}).
     * @throws NullPointerException
     * if {@code e} is {@code null}.
     * @throws IllegalArgumentException
     * if {@code e} does not have a correct message key which
     * matches this list's key.
     * @throws UnsupportedOperationException
     * if this list is marked as read-only.
     */
    @Override
    public boolean add(final E e)
        throws NullPointerException,
               IllegalArgumentException,
               UnsupportedOperationException
    {
        if (mReadOnlyFlag == true)
        {
            throw (
                new UnsupportedOperationException(
                    "list is read-only"));
        }
        else if (e == null)
        {
            throw (new NullPointerException("e is null"));
        }
        else if ((e.key()).equals(mKey) == false)
        {
            throw (
                new IllegalArgumentException(
                    "e has incorrect message key"));
        }

        return (super.add(e));
    } // end of add(EMessage)

    /**
     * Inserts the specified element at the specified position in
     * this list. Shifts the element currently at that position
     * (if any) and any subsequent elements to the right (adds
     * one to their indices).
     * @param index insert {@code e} into list at this index.
     * @param e insert this element into list.
     * @throws NullPointerException
     * if {@code e} is {@code null}.
     * @throws IllegalArgumentException
     * if {@code e} does not have a correct message key which
     * matches this list's key.
     * @throws IndexOutOfBoundsException
     * if {@code index} is out of range
     * ({@code index < 0 || index > this.size()}).
     * @throws UnsupportedOperationException
     * if this list is marked as read-only.
     */
    @Override
    public void add(final int index,
                    final E e)
        throws NullPointerException,
               IllegalArgumentException,
               IndexOutOfBoundsException
    {
        if (mReadOnlyFlag == true)
        {
            throw (
                new UnsupportedOperationException(
                    "list is read-only"));
        }
        else if (e == null)
        {
            throw (new NullPointerException("e is null"));
        }
        else if ((e.key()).equals(mKey) == false)
        {
            throw (
                new IllegalArgumentException(
                    "e has incorrect message key"));
        }

        super.add(index, e);

        return;
    } // end of add(int, EMessage)

    /**
     * Appends all of the elements in the specified collection to
     * the end of this list, in the order that they are returned
     * by the specified collection's Iterator. The behavior of
     * this operation is undefined if the specified collection is
     * modified while the operation is in progress. (This implies
     * that the behavior of this call is undefined if the
     * specified collection is this list, and this list is
     * nonempty.)
     * @param c append the elements of {@code c} to this list.
     * @return {@code true} if this list changed as a result of
     * the call.
     * @throws NullPointerException
     * if {@code c} is {@code null}.
     * @throws IllegalArgumentException
     * if {@code c} contains elements that are either
     * {@code null} or messages that have message keys that do
     * not match this list's message key.
     * @throws UnsupportedOperationException
     * if this list is marked as read-only.
     */
    @Override
    public boolean addAll(final Collection<? extends E> c)
        throws NullPointerException,
               IllegalArgumentException,
               UnsupportedOperationException
    {
        if (mReadOnlyFlag == true)
        {
            throw (
                new UnsupportedOperationException(
                    "list is read-only"));
        }
        else if (c == null)
        {
            throw (new NullPointerException("c is null"));
        }

        validate(c, mKey);

        return (super.addAll(c));
    } // end of addAll(Collection<>)

    /**
     * Inserts all of the elements in the specified collection
     * into this list, starting at the specified position. Shifts
     * the element currently at that position (if any) and any
     * subsequent elements to the right (increases their
     * indices). The new elements will appear in the list in the
     * order that they are returned by the specified collection's
     * iterator.
     * @param index insert {@code c}'s elements into this list
     * at this index.
     * @param c collection containing insertion elements.
     * @return {@code true} if this list changed as a result of
     * the call.
     * @throws NullPointerException
     * if {@code c} is {@code null}.
     * @throws IllegalArgumentException
     * if {@code c} contains elements that are either
     * {@code null} or messages that have message keys that do
     * not match this list's message key.
     * @throws IndexOutOfBoundsException
     * if {@code index} is out of range
     * ({@code index < 0 || index > this.size()}).
     * @throws UnsupportedOperationException
     * if this list is marked as read-only.
     */
    @Override
    public boolean addAll(final int index,
                          final Collection<? extends E> c)
        throws NullPointerException,
               IllegalArgumentException,
               IndexOutOfBoundsException,
               UnsupportedOperationException
    {
        if (mReadOnlyFlag == true)
        {
            throw (
                new UnsupportedOperationException(
                    "list is read-only"));
        }
        else if (c == null)
        {
            throw (new NullPointerException("c is null"));
        }

        validate(c, mKey);

        return (super.addAll(index, c));
    } // end of addAll(int, Collection<>)

    /**
     * Replaces the element at the specified position in this
     * list with the specified element.
     * @param index replace element at this index.
     * @param e put this element at specified index.
     * @return the element previously at the specified position.
     * @throws NullPointerException
     * if {@code e} is {@code null}.
     * @throws IllegalArgumentException
     * if {@code e} does not have a correct message key which
     * matches this list's key.
     * @throws IndexOutOfBoundsException
     * if {@code index} is out of range
     * ({@code index < 0 || index > this.size()}).
     * @throws UnsupportedOperationException
     * if this list is marked as read-only.
     */
    @Override
    public E set(final int index, final E e)
        throws NullPointerException,
               IllegalArgumentException,
               IndexOutOfBoundsException,
               UnsupportedOperationException
    {
        if (mReadOnlyFlag == true)
        {
            throw (
                new UnsupportedOperationException(
                    "list is read-only"));
        }
        else if (e == null)
        {
            throw (new NullPointerException("e is null"));
        }
        else if ((e.key()).equals(mKey) == false)
        {
            throw (
                new IllegalArgumentException(
                    "e has incorrect message key"));
        }

        return (super.set(index, e));
    } // end of set(int, EMessage)

    //
    // end of EAbstractList Method Overrides.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get Methods.
    //

    /**
     * Returns the list element message key.
     * @return message key.
     */
    public EMessageKey key()
    {
        return (mKey);
    } // end of key()

    //
    // end of Get Methods.
    //-----------------------------------------------------------

    /**
     * Validates that the message elements in the given
     * collection are not {@code null} and have a key matching
     * {@code key}. This method is called for effect only.
     * @param c validate this message collections elements.
     * @param key message elements must match this key.
     * @throws IllegalArgumentException
     * if {@code c} contains either {@code null} elements or
     * messages with the incorrect key.
     */
    private static void validate(final Collection<? extends EMessage> c,
                                 final EMessageKey key)
        throws IllegalArgumentException
    {
        c.forEach(
            msg ->
            {
                if (msg == null)
                {
                    throw (
                        new IllegalArgumentException(
                            "collection contains null message"));
                }
                else if ((msg.key()).equals(key) == false)
                {
                    throw (
                        new IllegalArgumentException(
                            "collection contains message with incorrect key"));
                }
            });

        return;
    } // end of validate(Collection<>, EMessageKey)
} // end of class EMessageList
