//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2011 - 2013, 2015, 2016. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.messages;

import java.net.SocketAddress;

/**
 * Contains the {@link EMessage} header information. This
 * includes the "from" proxy identifier, the "to" proxy
 * identifier, and the eBus {@link EMessage}. The proxy
 * identifiers are used to route messages between remote eBus
 * applications. The proxies are used to represent the remote
 * eBus' capability within the local JVM.
 * <p>
 * This class is immutable.
 * <p>
 * The binary serialization format for an eBus message header is:
 * <ul>
 *   <li>
 *     2-byte, signed integer serialized header length. These two
 *     bytes are also used for the heartbeat (-15,000) and
 *     heartbeat reply (-8,000) indicators.
 *   </li>
 *   <li>
 *     4-byte, signed integer message class identifier. This is
 *     the same identifier received from the remote application's
 *     {@link net.sf.eBus.client.sysmessages.KeyMessage}.
 *   </li>
 *   <li>
 *     4-byte, signed integer from feed identifier. The local
 *     eBus feed responsible for sending this message.
 *   </li>
 *   <li>
 *     4-byte, signed integer to feed identifier. The remote
 *     eBus feed responsible for handling this message. If the
 *     remote feed is not known when the message is sent, it is
 *     set to {@link #NO_ID}.
 *   </li>
 *   <li>
 *     The de-serialized eBus message. See
 *     {@link net.sf.eBus.messages.type.MessageType} about eBus
 *     message and field serialization.
 *   </li>
 * </ul>
 * <p>
 * Both the "to" and "from" feeds are encapsulated in an
 * {@code ERemoteApp} instance.
 * </p>
 *
 * @see net.sf.eBus.messages.EMessage
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public final class EMessageHeader
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * Proxy identifiers are set to -1 to signify that they are
     * unknown at the time the message was sent.
     */
    public static final int NO_ID = -1;

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * The unique class identifier. This identifier
     * is unique within the scope of its local eBus application.
     */
    private final int mClassId;

    /**
     * The message was sent from this eBus proxy instance. Set to
     * {@link #NO_ID} for system messages.
     */
    private final int mFromFeedId;

    /**
     * The message is destined for this eBus proxy instance. Set
     * to {@link #NO_ID} if the destination proxy is unknown.
     */
    private final int mToFeedId;

    /**
     * The optional source or destination address for this eBus
     * message. This field is only set for messages posted via
     * UDP datagram packets. Set to {@code null} when posting
     * messages via TCP.
     */
    private final SocketAddress mAddress;

    /**
     * The encapsulated eBus message.
     */
    private final EMessage mMessage;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a message header with the given key, from and to
     * feed identifiers and eBus message. Sets the message's
     * associated address to {@code null}.
     * @param classId the message class identifier.
     * @param fromFeedId the message is from this eBus proxy.
     * Will be {@link #NO_ID} for
     * {@link ESystemMessage system messages} only; all other
     * messages must set this to a value &ge; zero.
     * @param toFeedId the message is destined for this eBus
     * proxy. Will be {@link #NO_ID} if the destination proxy is
     * unknown.
     * @param msg the eBus message.
     * @throws IllegalArgumentException
     * if {@code msg} is {@code null} or if {@code msg} is an
     * application message and {@code classId} or
     * {@code fromFeedId} is &lt; zero.
     */
    public EMessageHeader(final int classId,
                          final int fromFeedId,
                          final int toFeedId,
                          final EMessage msg)
    {
        this (classId, fromFeedId, toFeedId, null, msg);
    } // end of EMessageHeader(int, int, int, EMessage)

    /**
     * Creates a message header with the given key, from and to
     * feed identifiers and eBus message.
     * @param classId the message class identifier.
     * @param fromFeedId the message is from this eBus proxy.
     * Will be {@link #NO_ID} for
     * {@link ESystemMessage system messages} only; all other
     * messages must set this to a value &ge; zero.
     * @param toFeedId the message is destined for this eBus
     * proxy. Will be {@link #NO_ID} if the destination proxy is
     * unknown.
     * @param address message source or destination address. May
     * be {@code null}.
     * @param msg the eBus message.
     * @throws IllegalArgumentException
     * if {@code msg} is {@code null} or if {@code msg} is an
     * application message and {@code classId} or
     * {@code fromFeedId} is &lt; zero.
     */
    public EMessageHeader(final int classId,
                          final int fromFeedId,
                          final int toFeedId,
                          final SocketAddress address,
                          final EMessage msg)
    {
        if (msg == null)
        {
            throw (new IllegalArgumentException("null msg"));
        }
        // If this is a system message, then the class and from
        // feed identifiers may be less than zero.
        else if (msg.isApplicationMessage() == false)
        {
            // no-op
        }
        else if (classId < 0)
        {
            throw (new IllegalArgumentException("classId < 0"));
        }
        else if (fromFeedId < 0)
        {
            throw (
                new IllegalArgumentException("fromProxyId < 0"));
        }

        mClassId = classId;
        mFromFeedId = fromFeedId;
        mToFeedId = toFeedId;
        mAddress = address;
        mMessage = msg;
    } // end of EMessageHeader(...)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get methods
    //

    /**
     * Returns the message class.
     * @return message class.
     */
    public Class<? extends EMessage> messageClass()
    {
        return (mMessage.getClass());
    } // end of messageClass()

    /**
     * Returns {@code true} if the contained message is a
     * system message.
     * @return {@code true} if a system message.
     */
    public boolean isSystemMessage()
    {
        return (mMessage.messageType() ==
                    EMessage.MessageType.SYSTEM);
    } // end of isSystemMessage()

    /**
     * Returns the encapsulated message type.
     * @return message type.
     */
    public EMessage.MessageType messageType()
    {
        return (mMessage.messageType());
    } // end of messageType()

    /**
     * Returns the encapsulated message key.
     * @return message key.
     */
    public EMessageKey messageKey()
    {
        return (mMessage.key());
    } // end of messageKey()

    /**
     * Returns the unique class identifier.
     * @return class identifier.
     */
    public int classId()
    {
        return (mClassId);
    } // end of classId()

    /**
     * Returns the from proxy identifier. If this is a
     * {@link ESystemMessage}, then returns {@link #NO_ID}.
     * @return the from proxy identifier.
     */
    public int fromFeedId()
    {
        return (mFromFeedId);
    } // end of fromFeedId()

    /**
     * Returns the to proxy identifier. If the destination proxy
     * is not known at the time the message was sent, then
     * returns {@link #NO_ID}.
     * @return the destination proxy identifier.
     */
    public int toFeedId()
    {
        return (mToFeedId);
    } // end of toFeedId()

    /**
     * Returns the message source or destination address. May
     * return {@code null}.
     * @return socket address.
     */
    public SocketAddress address()
    {
        return (mAddress);
    } // end of address()

    /**
     * Returns the encapsulated eBus message.
     * @return the encapsulated eBus message.
     */
    public EMessage message()
    {
        return (mMessage);
    } // end of message()

    //
    // end of Get methods.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Object Method Overrides.
    //

    /**
     * Returns this header as human-readable text.
     * @return the header as text.
     */
    @Override
    public String toString()
    {
        return (String.format("[key=%d,from=%d;to=%d]%n%s",
                              mClassId,
                              mFromFeedId,
                              mToFeedId,
                              mMessage));
    } // end of toString()

    //
    // end of Object Method Overrides.
    //-----------------------------------------------------------
} // end of class EMessageHeader
