//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright (C) 2001 - 2011, 2013, 2016, 2019. Charles W. Rapp.
// All Rights Reserved.
//

package net.sf.eBus.messages;

import com.google.common.base.Strings;
import java.io.Serializable;
import java.time.Instant;
import java.util.List;

/**
 * {@code EMessage} is the base class for all eBus messages and
 * contains the message subject and timestamp.
 * Application messages do <em>not</em> extend this class
 * directly but one of the subclasses:
 * {@link ENotificationMessage}, {@link ERequestMessage}, or
 * {@link EReplyMessage}.
 * <p>
 * <strong>Note:</strong> messages are limited to 31 fields. This
 * is due to eBus binary serialization format. This number is
 * decreased to 29 fields for sub-classes because
 * {@code EMessage} used two fields: {@link #subject} and
 * {@link #timestamp}. The work-around to this limitation is to
 * group fields into an {@link EField} subclass, which is allowed
 * to have 31 fields of its own. So if a message class uses two
 * {@code EField} subclasses, it may contain up to 62 fields with
 * 31 in each field subclass.
 * </p>
 * <p>
 * <strong>Note:</strong> a {@code EMessage} subclass may
 * be used as a message field itself. That means
 * that a message may contain another message as a field.
 * </p>
 * <p>
 * This class is immutable.
 * </p>
 * <p>
 * See {@link EMessageObject} for detailed explanation of the
 * required builder inner class.
 * </p>
 *
 * @see EMessageHeader
 * @see EMessageObject
 * @see ENotificationMessage
 * @see ERequestMessage
 * @see EReplyMessage
 * @see EField
 *
 * @author <a href="mailto:crapp@eqtc.com">Charles Rapp</a>
 */

public abstract class EMessage
    extends EMessageObject
    implements Serializable
{
//---------------------------------------------------------------
// Enums.
//

    /**
     * Messages are divided into four types: notification,
     * request, reply and system. System messages are reserved
     * for use by the eBus system itself and may not be used
     * by applications.
     */
    public enum MessageType
    {
        /**
         * A notification is an event with an associated
         * subject. Notifications are routed by subject.
         */
        NOTIFICATION (ENotificationMessage.class),

        /**
         * A request contains a subject and request identifier
         * and is routed by subject.
         */
        REQUEST (ERequestMessage.class),

        /**
         * A reply contains a subject, request identifier,
         * reply status and optional reason. Replies are routed
         * by their destination address. The destination is
         * the corresponding request source address.
         */
        REPLY (EReplyMessage.class),

        /**
         * eBus system message. Reserved for use by the eBus
         * system and may not be used by applications.
         */
        SYSTEM (ESystemMessage.class);

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        private MessageType(final Class<? extends EMessage> mc)
        {
            mMsgClass = mc;
        } // end of MessageType(Class)

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Get Methods.
        //

        /**
         * Returns the associated message class.
         * @return message class.
         */
        public Class<? extends EMessage> messageClass()
        {
            return (mMsgClass);
        } // end of messageClass()

        /**
         * Returns {@code true} if {@code mc} is not {@code null}
         * and is assignable from this message type.
         * @param mc check if this message class matches this
         * type.
         * @return {@code true} if {@code mc} matches this
         * message type.
         */
        public boolean isMatching(final Class mc)
        {
            return (mc != null && mMsgClass.isAssignableFrom(mc));
        } // end of isMatching(Class)

        //
        // end of Get Methods.
        //-------------------------------------------------------

    //-----------------------------------------------------------
    // Member data.
    //

        /**
         * Message class associated with this message type.
         */
        private final Class<? extends EMessage> mMsgClass;
    } // end of enum MessageType

//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * Serialization version identifier.
     */
    private static final long serialVersionUID = 0x050200L;

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * The required message subject. The subject and the message
     * class are used to route this message to the intended
     * eBus clients.
     */
    public final String subject;

    /**
     * The message timestamp in Java millisecond epoch time.
     */
    public final long timestamp;

    /**
     * The message type. Either notification, request, reply, or
     * system.
     */
    private final MessageType mMessageType;

    /**
     * The message key combining the message class and subject.
     */
    private final EMessageKey mKey;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new eBus message for the given subject and
     * timestamp. The message type defaults to
     * {@link MessageType#SYSTEM}.
     * @param subject the message timestamp.
     * @param timestamp the message timestamp in Java,
     * millisecond epoch time.
     * @throws IllegalArgumentException
     * if {@code subject} is either {@code null} or empty.
     * @throws InvalidMessageException
     * if this message violates the eBus correct message rules.
     *
     * @deprecated use {@link #EMessage(EMessage.Builder)} in the
     * future. System messages are not local-nly by definition
     * since they are used to communicate between eBus
     * applications.
     */
    @Deprecated
    public EMessage(final String subject,
                    final long timestamp)
        throws IllegalArgumentException,
               InvalidMessageException
    {
        this (subject, timestamp, MessageType.SYSTEM);
    } // end of EMessage(String, long)

    /**
     * Creates a new eBus message based on the given subject,
     * timestamp, and message type.
     * <p>
     * This constructor is <em>not</em> deprecated since it is
     * needed by local messages designated by
     * {@code &commat;ELocalOnly} annotation.
     * </p>
     * @param subject the message subject.
     * @param timestamp the message timestamp in Java,
     * millisecond epoch time.
     * @param msgType the message type.
     * @throws IllegalArgumentException
     * if:
     * <ul>
     *   <li>
     *     {@code subject} is either of {@code null} or empty;
     *   </li>
     *   <li>
     *     {@code msgType} is {@code null}.
     *   </li>
     * </ul>
     * @throws InvalidMessageException
     * if this message violates the eBus correct message rules.
     */
    /* package */ EMessage(final String subject,
                           final long timestamp,
                           final MessageType msgType)
        throws IllegalArgumentException,
               InvalidMessageException
    {
        if (subject == null || subject.isEmpty() == true)
        {
            throw (
                new IllegalArgumentException(
                    "null or empty subject"));
        }
        else if (msgType == null)
        {
            throw (new IllegalArgumentException("null msgType"));
        }

        this.subject = subject;
        this.timestamp = timestamp;
        this.mMessageType = msgType;
        this.mKey = new EMessageKey(this.getClass(), subject);
    } // end of EMessage(String, long, MessageType)

    /**
     * Creates a new eBus message based on the given message
     * builder. {@code builder} is guaranteed to contain a valid
     * message configuration at this point.
     * @param builder contains the eBus message configuration.
     */
    protected EMessage(final Builder<?, ?> builder)
    {
        this.subject = builder.mSubject;
        this.timestamp = builder.mTimestamp;
        this.mMessageType = builder.mType;
        this.mKey = new EMessageKey(this.getClass(), this.subject);
    } // end of EMessage(Builder)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Object Method Overrides.
    //

    /**
     * Returns {@code true} if {@code o} is a
     * non-{@code null EMessage} instance with a subject and
     * timestamp equal to {@code this EMessage} instance and
     * {@code false} otherwise.
     * @param o comparison object.
     * @return {@code true} if the message fields are equal
     * and {@code false} otherwise.
     */
    @Override
    public boolean equals(final Object o)
    {
        boolean retcode = (this == o);

        if (retcode == false && o instanceof EMessage)
        {
            final EMessage msg = (EMessage) o;

            retcode = (subject.equals(msg.subject) == true &&
                       timestamp == msg.timestamp);
        }

        return (retcode);
    } // end of equals(Object)

    /**
     * Returns the message header hash code.
     * @return the message header hash code.
     */
    @Override
    public int hashCode()
    {
        return ((subject.hashCode() * 37) +
                (Long.valueOf(timestamp)).hashCode());
    } // end of hashCode()

    /**
     * Returns the message subject and timestamp as a string.
     * @return the message as text.
     */
    @Override
    public String toString()
    {
        return (
            String.format(
                "%1$s:%2$s%n              timestamp: %3$tY/%3$tm/%3$td %3$tH:%3$tM:%3$tS.%3$tL",
                this.getClass(),
                subject,
                new java.util.Date(timestamp)));
    } // end of toString()

    //
    // end of Object Method Overrides.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get methods.
    //

    /**
     * Returns the unique message key based on the message class
     * and subject.
     * @return the unique message key based on the message class
     * and subject.
     */
    public EMessageKey key()
    {
        return (mKey);
    } // end of key()

    /**
     * Returns the
     * {@link net.sf.eBus.messages.EMessage.MessageType message type}.
     * @return the message type.
     */
    public MessageType messageType()
    {
        return (mMessageType);
    } // end of messageType()

    /**
     * Returns {@code true} if this is a system message and
     * {@code false} otherwise. Only eBus is allowed to transmit
     * a system message.
     * @return {@code true} if this is a system message and
     * {@code false} otherwise.
     */
    public boolean isSystemMessage()
    {
        return (mMessageType == MessageType.SYSTEM);
    } // end of isSystemMessage()

    /**
     * Returns {@code true} if this is an application
     * message and {@code false} otherwise.
     * @return {@code true} if this is an application
     * message and {@code false} otherwise.
     */
    public boolean isApplicationMessage()
    {
        return (mMessageType != MessageType.SYSTEM);
    } // end of isApplicationMessage()

    //
    // end of Get methods.
    //-----------------------------------------------------------

//---------------------------------------------------------------
// Inner classes.
//

    /**
     * Base class for all {@link EMessage} builders. Used by
     * eBus when de-serializing an encoded message back into the
     * target message object.
     *
     * @param <M> builds this target message class.
     * @param <B> message builder subclass. Needed to return the
     * correct builder type when setting fields. If this were not
     * the case, field chaining would not work.
     */
    @SuppressWarnings ("unchecked")
    public static abstract class Builder<M extends EMessage,
                                         B extends Builder<M, ?>>
        extends EMessageObject.Builder<M, B>
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        /**
         * Message subject.
         */
        private String mSubject;

        /**
         * Message timestamp.
         */
        private long mTimestamp;

        /**
         * Message type.
         */
        private final MessageType mType;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * Creates an eBus message builder instance.
         * @param targetClass builds this message class.
         * @param type notification, request, or reply message.
         */
        protected Builder(final Class<? extends EMessageObject> targetClass,
                          final MessageType type)
        {
            this (targetClass, null, Long.MIN_VALUE, type);
        } // end of Builder(Class, MessageType)

        /**
         * Creates an eBus message builder instance for the given
         * message subject.
         * @param targetClass builds this message class.
         * @param subject message subject.
         * @param type notification, request, or reply message.
         */
        protected Builder(final Class<? extends EMessageObject> targetClass,
                          final String subject,
                          final MessageType type)
        {
            this(targetClass, subject, Long.MIN_VALUE, type);
        } // end of Builder(String, Class, MessageType)

        /**
         * Creates an eBus message builder instance for the given
         * message subject and timestamp.
         * @param timestamp message create timestamp.
         * @param subject message subject.
         * @param targetClass builds this message class.
         * @param type notification, request, or reply message.
         */
        protected Builder(final Class<? extends EMessageObject> targetClass,
                          final String subject,
                          final long timestamp,
                          final MessageType type)
        {
            super (targetClass);

            mSubject = subject;
            mTimestamp = timestamp;
            mType = type;
        } // end of Builder(String, long, Class, MessageType)

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Abstract Method Declarations.
        //

        /**
         * Checks if message subject and message type are
         * configured. If not, then text explaining this error
         * is appended to {@code problems}.
         * <p>
         * This method should be overridden by subclass message
         * builders and called <em>before</em> doing its own
         * validation. The first line in the subclass
         * {@code validate} implementation should be
         * {@code super.validate(problems);}.
         * </p>
         * <p>
         * When overriding this method, be sure to add all
         * discovered validation problems to the list. The
         * validation method should consist of a series of
         * individual {@code if} statements and <em>not</em>
         * an {@code if/else if} chain. That way all problems
         * are found and not just the first one.
         * </p>
         * @param problems append validation problems to this
         * list.
         */
        @Override
        protected void validate(final List<String> problems)
        {
            if (mSubject == null)
            {
                problems.add("subject not set");
            }

            if (mType == null)
            {
                problems.add("message type not set");
            }

            return;
        } // end of validate(List<>)

        //
        // end of Abstract Method Declarations.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        /**
         * Sets the message subject to the given value.
         * Returns {@code this} builder so that field
         * configuration may be chained.
         * @param subject message subject.
         * @return {@code this} builder instance.
         * @throws IllegalArgumentException
         * if {@code subject} is either {@code null} or
         * empty.
         */
        public final B subject(final String subject)
        {
            if (Strings.isNullOrEmpty(subject))
            {
                throw (
                    new IllegalArgumentException(
                        "subject is null or empty"));
            }

            mSubject = subject;

            return ((B) this);
        } // end of subject(String)

        /**
         * Sets the message timestamp to the given value.
         * Returns {@code this} builder so that field
         * configuration may be chained.
         * @param ts message timestamp.
         * @return {@code this} builder instance.
         */
        public final B timestamp(final long ts)
        {
            mTimestamp = ts;

            return ((B) this);
        } // end of timestamp(long)

        /**
         * Sets the message timestamp to the given value.
         * Returns {@code this} builder so that field
         * configuration may be chained.
         * @param ts message timestamp.
         * @return {@code this} builder instance.
         */
        public final B timestamp(final Instant ts)
        {
            mTimestamp = ts.toEpochMilli();

            return ((B) this);
        } // end of timestamp(final Instant ts)

        //
        // end of Set Methods.
        //-------------------------------------------------------
    } // end of class Builder<M, B>
} // end of class EMessage
