/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later
 * version.
 *
 * This library is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General
 * Public License along with this library; if not, write to the
 *
 * Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330,
 * Boston, MA
 * 02111-1307 USA
 *
 * The Initial Developer of the Original Code is Charles W. Rapp.
 * Portions created by Charles W. Rapp are
 * Copyright 2019. Charles W. Rapp
 * All Rights Reserved.
 */

package net.sf.eBus.messages;

import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * This class-level annotation is used to mark an
 * {@link EMessage} or {@link EField} class as local-only meaning
 * that the message may only be used within the JVM and will not
 * be transmitted to remote eBus applications. Marking a message
 * as {@code @ELocalOnly} prevents eBus from generating the
 * serialization/de-serialization code necessary for message
 * transmission. This means that a local-only message may
 * contain unsupported field types like {@link java.util.Map}.
 * <p>
 * This annotation should be used when messages are designed for
 * internal use only and should not be transmitted outside the
 * JVM. {@code net.sf.eBusx.io}, {@code net.sf.eBusx.util}, and
 * {@code net.sf.eBus.feed} packages use this feature for their
 * JVM-only messages.
 * </p>
 * <p>
 * Note that a message is assumed to transmittable unless
 * annotated as {@code ELocalOnly}.
 * </p>
 * <p>
 * It is an error to open an eBus feed for a {@code ELocalOnly}
 * message but with {@link net.sf.eBus.client.EFeed.FeedScope}
 * other than {@code FeedScope.LOCAL_ONLY}.
 * </p>
 * <p>
 * It is an error to define an eBus message or field as
 * transmittable (not local-only) but contains a local-only
 * {@link EMessageObject} field.
 * </p>
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

@Retention(RetentionPolicy.RUNTIME)
@Target(ElementType.TYPE)
@Documented
public @interface ELocalOnly
{}
