//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2017. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.messages;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;

/**
 * An {@link ArrayList} specifically containing {@link EField}
 * elements. The difference between this class and
 * {@code EField[]} is that this list contains homogenous field
 * elements: all the fields must belong to the same
 * <em>concrete</em> class (that is, a non-abstract class).
 * Homogeneity decreases the serialized buffer size and decreases
 * the serialization/de-serialization processing time. The reason
 * is that the listed {@code EField} class is serialized once and
 * its data type is used for all field elements. Contrast this
 * {@code EField[]} which allows heterogenous elements. Each
 * field class must be serialized and its associated data
 * type used to serialize the element. This greatly increases the
 * serialized buffer size and processing time.
 * <p>
 * In short, when transmitting a homogenous {@code EField} list,
 * {@code EFieldList} is to be preferred over {@code EField[]}.
 * </p>
 * <p>
 * This class extends {@link ArrayList}, providing all the
 * features available in that class.
 * </p>
 * <p>
 * There is no limit as to the number of elements which may be
 * placed into this list. But care must be taken not to overflow
 * the eBus {@code ByteBuffer} when serializing for transport.
 * </p>
 * <p>
 * Java allows {@code EFieldList} to be instantiated without
 * using the generic parameter. That allows non-{@code EField}
 * objects to be placed into the list. Another problem is using
 * an abstract {@code EField} class as the generic parameter.
 * <strong>{@code EFieldList} does not protect against these uses,
 * resulting in undefined and unwanted behavior.</strong>
 * {@code EFieldList} should be declared with a concrete
 * {@code EField} subclass.
 * </p>
 *
 * @param <E> specifies the homogenous, concrete {@code EField}
 * type stored in this list.
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public final class EFieldList<E extends EField>
    extends EAbstractList<E>
    implements Serializable,
               Cloneable
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * Java serializable unique identifier.
     */
    private static final long serialVersionUID = 0x050200L;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates an empty field list instance with an initial
     * capacity of ten.
     */
    public EFieldList()
    {
        super ();
    } // end of EFieldList()

    /**
     * Creates an empty field list instance with the specified
     * initial capacity.
     * @param initialCapacity initial list capacity.
     */
    public EFieldList(final int initialCapacity)
    {
        super (initialCapacity);
    } // end of EFieldList(int)

    /**
     * Creates a field list instance containing the same elements
     * as {@code c} and in the order returned by the collection's
     * iterator.
     * @param c place this collection's element into this field
     * list.
     * @throws NullPointerException
     * if {@code c} is {@code null}.
     */
    public EFieldList(final Collection<E> c)
    {
        super (c);
    } // end of EFieldList(Collection<>)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get Methods.
    //

    /**
     * Returns the stored element class. Returns {@code null}
     * if the list is empty.
     * @return stored element class.
     */
    @SuppressWarnings ("unchecked")
    public Class<E> elementClass()
    {
        Class<E> retval = null;

        if (isEmpty() == false)
        {
            retval = (Class<E>) (get(0)).getClass();
        }

        return (retval);
    } // end of elementClass()

    //
    // end of Get Methods.
    //-----------------------------------------------------------
} // end of class EFieldList
