//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright (C) 2013, 2016, 2019. Charles W. Rapp.
// All Rights Reserved.
//

package net.sf.eBus.messages;

import java.io.Serializable;

/**
 * All application classes intended for use as an eBus messages
 * field must extend {@code EField}. This allows instances of
 * the application class to be serialized to a
 * {@link java.nio.ByteBuffer}. The application class must also
 * define {@code public static Builder builder()} and the
 * {@code Builder} inner class. See
 * <p>
 * <strong>Note:</strong> {@code EField} subclasses are limited
 * to 31 fields. This is due to eBus binary serialization format.
 * </p>
 * <h1>De-serialization</h1>
 * <p>
 * {@link EMessageObject} shows how {@code EMessage} and
 * {@code EField} need to define a builder inner class used to
 * construct a subclass. But the example assumed that the
 * subclass was {@code final}. This example shows how to define
 * builders for a non-{@code final EField} class. The example
 * also applies to {@code EMessage}. The following examples
 * show two classes: {@code UserInfo} and its subclass
 * {@code Employee}.
 * </p>
 * <pre><code>public class UserInfo
    extends EField
{
    public final String name;
    public final int age;

    private UserInfo(final Builder&lt;?, ?&gt; builder) {
        super (builder);

        this.name = builder.mName;
        this.age = builder.mAge;
    }

    <em>// builder code explained below.</em>
}

public final class Employee
    extends UserInfo
{
    public final String department;

    private Employee(final Builder builder) {
        super (builder);

        this.department = builder.mDepartment;
    }

    <em>// builder code explained below.</em>
}</code></pre>
 * <p>
 * Class {@code UserInfo} can be instantiated on its own or as
 * {@code Employee} superclass. Because of this {@code UserInfo}
 * is required to have two builder inner classes. The first is
 * an abstract builder which the {@code Employee} builder
 * extends. The second is a concrete builder used to create
 * {@code UserInfo} instances.
 * </p>
 * <h2>UserInfo abstract builder</h2>
 * <pre><code>public static abstract class Builder&lt;M extends UserInfo,
                                     B extends Builder&lt;M, ?&gt;&gt;
    extends EField.Builder&lt;M, B&gt;
{
    private String mName;
    private int mAge;

    <em>// Protected constructor to allow subclass access.</em>
    protected Builder(final Class&lt;? extends EMessageObject&gt; targetClass) {
        super (targetClass);

        mAge = -1;
    }

    public B name(final String value)
    {
        if (name == null || name.isEmpty()) {
            throw (new IllegalArgumentException("value is null or empty"));
        }

        mName = value;
        return ((B) this);
    }

    public B age(final int value)
    {
        if (age &lt; 0) {
            throw (new IllegalArgumentException("value &lt; zero"));
        }

        mAge = age;
        return ((B) this);
    }

    &#64;Override protected void validate(final List&lt;String&gt; problems)
    {
        super.validate(problems);

        if (mName == null) { problems.add("name not set"); }
        if (mAge &lt; 0) { problems.add("age not set"); }
    }

    <em>// Since this class is abstract do not override buildImpl method.</em>
}</code></pre>
 * <p>
 * Since {@code Builder} is abstract it must use have the two
 * class parameters {@code M} and {@code B} which are then passed
 * to the superclass builder. Note that the setters use {@code B}
 * in the method signature so that the subclass type is returned
 * rather than {@code UserInfo.Builder}. Method {@code buildImpl}
 * is not defined since that is only done in concrete builders.
 * </p>
 * <h2>UserInfo concrete builder</h2>
 * <p>
 * {@code UserInfo.ConcreteBuilder} extends
 * {@code UserInfo.Builder} and defines {@code buildImpl}. There
 * is nothing else for {@code ConcreteBuilder} to do since the
 * setters and validation methods are provided by the superclass.
 * </p>
 * <pre><code>public static final class ConcreteBuilder
    extends Builder&lt;UserInfo, ConcreteBuilder&gt;
{
    private ConcreteBuilder() {
        super (UserInfo.class);
    }

    &#64;Override protected UserInfo buildImpl()
    {
        return (new UserInfo(this));
    }
}</code></pre>
 * <h2>UserInfo builder method</h2>
 * <p>
 * {@code UserInfo.builder} method returns a
 * {@code UserInfo.ConcreteBuilder} instance since that class is
 * responsible for creating {@code UserInfo} instances. But the
 * return type is the abstract {@code UserInfo.Builder}.
 * </p>
 * <pre><code>public static Builder builder() {
    return (new ConcreteBuilder());
}</code></pre>
 * <h2>Employee builder</h2>
 * <p>
 * {@code Employee.Builder} is as you expect: it extends
 * {@code UserInfo.Builder}, defines the {@code department}
 * setter method, validates the builder configuration, and
 * builds the {@code Employee} instance.
 * </p>
 * <pre><code>public static final class Builder
    extends UserInfo.Builder&lt;Employee, Builder&gt;
{
    private String mDepartment;

    private Builder() {
        super (Employee.class);
    }

    public Builder department(final String value) {
        if (value == null || value.isEmpty()) {
            throw (new IllegalArgumentException("value is null or empty"));
        }

        mDepartment = value;
        return (this);
    }

    &#64;Override protected void validate(List&lt;String&gt; problems) {
        super.validate(problems);

        if (mDepartment == null) { problems.add("department not set"); }
    }

    &#64;Override protected Employee buildImpl() {
        return (new Employee(this));
    }
}</code></pre>
 *
 * @see EMessageObject
 * @see EMessage
 *
 * @author <a href="mailto:crapp@eqtc.com">Charles Rapp</a>
 */

public abstract class EField
    extends EMessageObject
    implements Serializable
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * Serialization version identifier.
     */
    private static final long serialVersionUID = 0x050200L;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new eBus field instance. This construct is
     * public because the
     * {@link net.sf.eBus.messages.type.MessageType} requires
     * access.
     * @throws InvalidMessageException
     * if this field violates the eBus correct field rules.
     */
    public EField()
        throws InvalidMessageException
    {}

    protected EField(final Builder<?, ?> builder)
    {
        super (builder);
    } // end of EField(Builder<>)

    //
    // end of Constructors.
    //-----------------------------------------------------------

//---------------------------------------------------------------
// Inner classes.
//

    public static abstract class Builder<M extends EField,
                                         B extends Builder<M, ?>>
        extends EMessageObject.Builder<M, B>
    {
    //-----------------------------------------------------------
    // Member data.
    //

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        protected Builder(final Class<? extends EMessageObject> targetClass)
        {
            super (targetClass);
        } // end of Builder(Class)

        //
        // end of Constructors.
        //-------------------------------------------------------
    } // end of class Builder<M, B>
} // end of class EField
