//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2017. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.messages;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.function.UnaryOperator;


/**
 * Base class for {@link EFieldList} and {@link EMessageList}
 * classes, implementing the read-only functionality.
 *
 * @param <E> specifies the {@link EMessageObject} type stored in
 * this list.
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public abstract class EAbstractList<E extends EMessageObject>
    extends ArrayList<E>
    implements Serializable,
               Cloneable
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * Java serializable unique identifier.
     */
    private static final long serialVersionUID = 0x050200L;

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * When set to {@code true} list may only be read from and
     * not updated.
     */
    protected boolean mReadOnlyFlag;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates an empty field list instance with an initial
     * capacity of ten.
     */
    protected EAbstractList()
    {
        super ();

        mReadOnlyFlag = false;
    } // end of EAbstractList()

    /**
     * Creates an empty list instance with the specified
     * initial capacity.
     * @param initialCapacity initial list capacity.
     */
    protected EAbstractList(final int initialCapacity)
    {
        super (initialCapacity);

        mReadOnlyFlag = false;
    } // end of EAbstractList(int)

    /**
     * Creates a list instance containing the same elements
     * as {@code c} and in the order returned by the collection's
     * iterator.
     * @param c place this collection's element into this field
     * list.
     * @throws NullPointerException
     * if {@code c} is {@code null}.
     */
    protected EAbstractList(final Collection<E> c)
    {
        super (c);

        mReadOnlyFlag = false;
    } // end of EAbstractList(Collection)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // ArrayList Method Overrides.
    //

    @Override
    public void sort(Comparator<? super E> c)
    {
        if (mReadOnlyFlag == true)
        {
            throw (
                new UnsupportedOperationException(
                    "list is read-only"));
        }

        super.sort(c);

        return;
    } // end of sort(Comparator)

    @Override
    public List<E> subList(final int fromIndex,
                           final int toIndex)
    {
        throw (
            new UnsupportedOperationException(
                    "subList operation is not supported"));
    } // end of subList(int, int)

    @Override
    public Iterator<E> iterator()
    {
        return (new EIterator<>(super.iterator()));
    } // end of iterator()

    @Override
    public ListIterator<E> listIterator()
    {
        return (new EListIterator<>(super.listIterator()));
    } // end of listIterator()

    @Override
    public ListIterator<E> listIterator(final int index)
    {
        return (new EListIterator<>(super.listIterator(index)));
    } // end of listIterator(int)

    @Override
    public void add(final int index,
                    final E element)
    {
        if (mReadOnlyFlag == true)
        {
            throw (
                new UnsupportedOperationException(
                    "list is read-only"));
        }

        super.add(index, element);

        return;
    } // end of add(int, E)

    @Override
    public boolean add(final E e)
    {
        if (mReadOnlyFlag == true)
        {
            throw (
                new UnsupportedOperationException(
                    "list is read-only"));
        }

        return (super.add(e));
    } // end of add(E)

    @Override
    public boolean addAll(final int index,
                          final Collection<? extends E> c)
    {
        if (mReadOnlyFlag == true)
        {
            throw (
                new UnsupportedOperationException(
                    "list is read-only"));
        }

        return (super.addAll(index, c));
    } // end of addAll(int, Collection<>)

    @Override
    public boolean addAll(final Collection<? extends E> c)
    {
        if (mReadOnlyFlag == true)
        {
            throw (
                new UnsupportedOperationException(
                    "list is read-only"));
        }

        return (super.addAll(c));
    } // end of addAll(Collection<>)

    @Override
    public E set(final int index,
                 final E element)
    {
        if (mReadOnlyFlag == true)
        {
            throw (
                new UnsupportedOperationException(
                    "list is read-only"));
        }

        return (super.set(index, element));
    } // end of set(int, E)

    @Override
    public boolean retainAll(final Collection<?> c)
    {
        if (mReadOnlyFlag == true)
        {
            throw (
                new UnsupportedOperationException(
                    "list is read-only"));
        }

        return (super.retainAll(c));
    } // end of retainAll(Collection)

    @Override
    public void replaceAll(final UnaryOperator<E> operator)
    {
        if (mReadOnlyFlag == true)
        {
            throw (
                new UnsupportedOperationException(
                    "list is read-only"));
        }

        super.replaceAll(operator); //To change body of generated methods, choose Tools | Templates.
    } // end of replaceAll(UnarayOperator)

    @Override
    public boolean remove(final Object o)
    {
        if (mReadOnlyFlag == true)
        {
            throw (
                new UnsupportedOperationException(
                    "list is read-only"));
        }

        return (super.remove(o));
    } // end of remove(Object)

    @Override
    public E remove(final int index)
    {
        if (mReadOnlyFlag == true)
        {
            throw (
                new UnsupportedOperationException(
                    "list is read-only"));
        }

        return (super.remove(index));
    } // end of remove(int)

    @Override
    public boolean removeAll(final Collection<?> c)
    {
        if (mReadOnlyFlag == true)
        {
            throw (
                new UnsupportedOperationException(
                    "list is read-only"));
        }

        return (super.removeAll(c));
    } // end of removeAll(Collection)

    @Override
    protected void removeRange(final int fromIndex,
                               final int toIndex)
    {
        if (mReadOnlyFlag == true)
        {
            throw (
                new UnsupportedOperationException(
                    "list is read-only"));
        }

        super.removeRange(fromIndex, toIndex);

        return;
    } // end of removeRange(int, int)

    @Override
    public boolean removeIf(final Predicate<? super E> filter)
    {
        if (mReadOnlyFlag == true)
        {
            throw (
                new UnsupportedOperationException(
                    "list is read-only"));
        }

        return (super.removeIf(filter));
    } // end of removeIf(Predicate)

    @Override
    public void clear()
    {
        if (mReadOnlyFlag == true)
        {
            throw (
                new UnsupportedOperationException(
                    "list is read-only"));
        }

        super.clear(); //To change body of generated methods, choose Tools | Templates.
    }

    //
    // end of ArrayList Method Overrides.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get Methods.
    //

    /**
     * Returns {@code true} if this field list is read-only,
     * meaning this list may not be modified.
     * @return {@code true} if this field list is read-only.
     */
    public final boolean isReadOnly()
    {
        return (mReadOnlyFlag);
    } // end of isReadOnly()

    //
    // end of Get Methods.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Set Methods.
    //

    /**
     * Marks this list as read-only. Once set, this list can
     * never again be modified, whether directly, via sublists,
     * or iterators.
     */
    public final void setReadOnly()
    {
        mReadOnlyFlag = true;
        return;
    } // end of setReadOnly()

    //
    // end of Set Methods.
    //-----------------------------------------------------------

//---------------------------------------------------------------
// Inner classes.
//

    /**
     * Encapsulates an {@code ArrayList} iterator. Check the
     * {@link #mReadOnlyFlag} before allowing element removal.
     * @param <E> the list element type.
     */
    private final class EIterator<E extends EMessageObject>
        implements Iterator<E>
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        private final Iterator<E> mRealIterator;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        private EIterator(final Iterator<E> it)
        {
            mRealIterator = it;
        } // end of EIterator(Iterator)

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Iterator Interface Implementation.
        //

        @Override
        public boolean hasNext()
        {
            return (mRealIterator.hasNext());
        } // end of hasNext()

        @Override
        public E next()
        {
            return (mRealIterator.next());
        } // end of next()

        @Override
        public void remove()
        {
            if (mReadOnlyFlag == true)
            {
                throw (
                    new UnsupportedOperationException(
                        "list is read-only"));
            }

            mRealIterator.remove();

            return;
        } // end of remove()

        @Override
        public void forEachRemaining(final Consumer<? super E> a)
        {
            mRealIterator.forEachRemaining(a);
            return;
        } // end of forEachRemaining(Consumer<>)

        //
        // end of Iterator Interface Implementation.
        //-------------------------------------------------------
    } // end of class EIterator

    /**
     * Encapsulates an {@code ArrayList} list iterator. Check the
     * {@link #mReadOnlyFlag} before allowing element addition or
     * removal.
     * @param <E> the list element type.
     */
    private final class EListIterator<E extends EMessageObject>
        implements ListIterator<E>
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        private final ListIterator<E> mRealIterator;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        private EListIterator(final ListIterator<E> lit)
        {
            mRealIterator = lit;
        } // end of EListIterator(ListIterator<>)

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // ListIterator Interface Implementation.
        //

        @Override
        public boolean hasNext()
        {
            return (mRealIterator.hasNext());
        } // end of hasNext()

        @Override
        public E next()
        {
            return (mRealIterator.next());
        } // end of next()

        @Override
        public boolean hasPrevious()
        {
            return (mRealIterator.hasPrevious());
        } // end of hasPrevious()

        @Override
        public E previous()
        {
            return (mRealIterator.previous());
        } // end of previous()

        @Override
        public int nextIndex()
        {
            return (mRealIterator.nextIndex());
        } // end of nextIndex()

        @Override
        public int previousIndex()
        {
            return (mRealIterator.previousIndex());
        } // end of previousIndex()

        @Override
        public void remove()
        {
            if (mReadOnlyFlag == true)
            {
                throw (
                    new UnsupportedOperationException(
                        "list is read-only"));
            }

            mRealIterator.remove();

            return;
        } // end of remove()

        @Override
        public void set(final E e)
        {
            if (mReadOnlyFlag == true)
            {
                throw (
                    new UnsupportedOperationException(
                        "list is read-only"));
            }

            mRealIterator.set(e);

            return;
        } // end of set(E)

        @Override
        public void add(E e)
        {
            if (mReadOnlyFlag == true)
            {
                throw (
                    new UnsupportedOperationException(
                        "list is read-only"));
            }

            mRealIterator.add(e);

            return;
        } // end of add(E)

        @Override
        public void forEachRemaining(final Consumer<? super E> a)
        {
            mRealIterator.forEachRemaining(a);
            return;
        } // end of forEachRemaining(Consumer<>)

        //
        // end of ListIterator Interface Implementation.
        //-------------------------------------------------------
    } // end of class EListIterator
} // end of class EAbstractList
