//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2012, 2013, 2015, 2016. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.client.sysmessages;

import java.io.Serializable;
import java.util.List;
import net.sf.eBus.client.EFeedState;

/**
 * This message is sent to a remote publisher to have a
 * notification feed either started or stopped.
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public final class SubscribeMessage
    extends AbstractKeyMessage
    implements Serializable
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * Serialization version identifier.
     */
    private static final long serialVersionUID = 0x050200L;

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * The subscription is either {@code EFeedState.UP} or
     * {@code EFeedState.DOWN}.
     */
    public final EFeedState feedState;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    private SubscribeMessage(final Builder builder)
    {
        super (builder);

        this.feedState = builder.mFeedState;
    } // end of SubscribeMessage(Builder)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Object Method Overrides.
    //

    /**
     * Returns {@code true} if {@code o} is a
     * non-{@code null SubscribeMessage} instance with the
     * same up flag as {@code this SubscribeMessage} instance
     * and {@code false} otherwise.
     * @param o comparison object.
     * @return {@code true} if the message fields are equal
     * and {@code false} otherwise.
     */
    @Override
    public boolean equals(final Object o)
    {
        boolean retcode = (this == o);

        if (retcode == false &&
            o instanceof SubscribeMessage)
        {
            final SubscribeMessage psm =
                (SubscribeMessage) o;

            retcode = (super.equals(o) == true &&
                       feedState == psm.feedState);
        }

        return (retcode);
    } // end of equals(Object)

    /**
     * Returns the feed status message hash code.
     * @return the feed status message hash code.
     */
    @Override
    public int hashCode()
    {
        return ((super.hashCode() * 37) + feedState.ordinal());
    } // end of hashCode()

    /**
     * Returns a human-readable text version of this message.
     * @return text version of this message.
     */
    @Override
    public String toString()
    {
        return (
            String.format("%s%n             feed state: %s",
                super.toString(),
                feedState));
    } // end of toString()

    //
    // end of Object Method Overrides.
    //-----------------------------------------------------------

    /**
     * Returns a new instance of the {@code SubscribeMessage}
     * builder.
     * @return message builder instance.
     */
    public static Builder builder()
    {
        return (new Builder());
    } // end of builder()

//---------------------------------------------------------------
// Inner classes.
//

    /**
     * Class used to create {@link SubscribeMessage} instances.
     * Used by eBus to de-serialize an encoded message.
     */
    public static final class Builder
        extends AbstractKeyMessage.Builder<SubscribeMessage, Builder>
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        private EFeedState mFeedState;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        private Builder()
        {
            super (SubscribeMessage.class);
        } // end of Builder()

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Builder Method Overrides.
        //

        /**
         * Returns the newly instantiated
         * {@code SubscribeMessage} based on this builder
         * configuration.
         * @return target message instance.
         */
        @Override
        protected SubscribeMessage buildImpl()
        {
            return (new SubscribeMessage(this));
        } // end of buildImpl()

        /**
         * Checks if the feed state is configured. If not, then
         * appends this problem to the list.
         * @param problems list of message configuration
         * problems.
         */
        @Override
        protected void validate(List<String> problems)
        {
            super.validate(problems);

            if (mFeedState == null)
            {
                problems.add("feed state is not set");
            }

            return;
        } // end of validate(List<>)

        //
        // end of Builder Method Overrides.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        /**
         * Sets the subscription feed state to the given value.
         * @param fs subscription feed state.
         * @return {@code this} builder.
         * @throws NullPointerException
         * if {@code fs} is {@code null}.
         */
        public Builder feedState(final EFeedState fs)
        {
            if (fs == null)
            {
                throw (new NullPointerException("fs is null"));
            }

            mFeedState = fs;

            return (this);
        } // end of feedState(EFeedState)

        //
        // end of Set Methods.
        //-------------------------------------------------------
    } // end of class Builder
} // end of class SubscribeMessage
