//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2019. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.client.sysmessages;

import java.io.Serializable;
import java.util.List;
import net.sf.eBus.messages.EReplyMessage.ReplyStatus;
import net.sf.eBus.messages.Optional;

/**
 * Response to a connection resume request. The reply status is
 * {@link ReplyStatus#OK_FINAL} when the request is accepted and
 * {@link ReplyStatus#ERROR} when rejected. If rejected, then
 * {@link #replyReason} contains text explaining the rejection.
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public final class ResumeReply
    extends AbstractLogonMessage
    implements Serializable
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * Serialization version identifier.
     */
    private static final long serialVersionUID = 0x050200L;

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * The current request handling status. If
     * {@link ReplyStatus#ERROR}, then {@link #replyReason} will
     * contain text explaining why the request was rejected.
     */
    public final ReplyStatus replyStatus;

    /**
     * The reason for a {@link ReplyStatus#ERROR} reply status.
     * May be {@code null} or empty.
     */
    @Optional
    public final String replyReason;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a resume reply message from the given message
     * configuration.
     * @param builder resume reply message builder.
     */
    private ResumeReply(final Builder builder)
    {
        super (builder);

        this.replyStatus = builder.mReplyStatus;
        this.replyReason = builder.mReplyReason;
    } // end of ResumeReply(Builder)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Object Method Overrides.
    //

    @Override
    public String toString()
    {
        return (
            String.format(
                "%s%n                 status: %s%n                reason: %s",
                super.toString(),
                replyStatus,
                replyReason));
    } // end of toString()

    //
    // end of Object Method Overrides.
    //-----------------------------------------------------------

    /**
     * Returns a new instance of the {@code ResumeReply} message
     * builder.
     * @return message builder instance.
     */
    public static Builder builder()
    {
        return (new Builder());
    } // end of builder()

//---------------------------------------------------------------
// Inner classes.
//

    /**
     * Class used to create {@link ResumeReply} messages. Used
     * by eBus to de-serialize an encoded message.
     */
    public static final class Builder
        extends AbstractLogonMessage.Builder<ResumeReply, Builder>
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        /**
         * The current request handling status. If
         * {@link ReplyStatus#ERROR}, then {@link #replyReason}
         * will contain text explaining why the request was
         * rejected.
         */
        private ReplyStatus mReplyStatus;

        /**
         * The reason for a {@link ReplyStatus#ERROR} reply status.
         * May be {@code null} or empty.
         */
        private String mReplyReason;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        private Builder()
        {
            super (ResumeReply.class);
        } // end of Builder()

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Builder Method Overrides.
        //

        /**
         * Returns the newly instantiated {@code ResumeReply}
         * message based on this builder configuration.
         * @return target message instance.
         */
        @Override
        protected ResumeReply buildImpl()
        {
            return (new ResumeReply(this));
        } // end of buildImpl()

        /**
         * Adds any and all configuration problems to the list.
         * @param problems list of mis-configurations.
         */
        @Override
        protected void validate(List<String> problems)
        {
            super.validate(problems);

            // Only reply status must be set.
            if (mReplyStatus == null)
            {
                problems.add("reply status not set");
            }

            return;
        } // end of validate(List<>)

        //
        // end of Builder Method Overrides.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        /**
         * Sets the reply status to the given value.
         * @param status reply status.
         * @return {@code this} resume reply builder.
         * @throws NullPointerException
         * if {@code status} is {@code null}.
         */
        public Builder replyStatus(final ReplyStatus status)
        {
            if (status == null)
            {
                throw (
                    new NullPointerException("status is null"));
            }

            mReplyStatus = status;

            return (this);
        } // end of replyStatus(ReplyStatus)

        /**
         * Sets the reply reason to the given value. May be
         * {@code null} or an empty string.
         * @param reason reply reason.
         * @return {@code this} resume reply builder.
         */
        public Builder replyReason(final String reason)
        {
            mReplyReason = reason;

            return (this);
        } // end of replyReason(String)

        //
        // end of Set Methods.
        //-------------------------------------------------------
    } // end of class Builder
} // end of class ResumeReply
