//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2015, 2016, 2019. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.client.sysmessages;

import java.io.Serializable;
import java.util.List;
import net.sf.eBus.messages.ESystemMessage;

/**
 * Acknowledges receipt of a request message from a remote eBus
 * application. This message is used to link the two feed
 * identifiers associated with the request. This allows a request
 * to be canceled before a reply is sent. This message has no
 * fields since no information beyond the header is required.
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public final class RemoteAck
    extends ESystemMessage
    implements Serializable
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * Serialization version identifier.
     */
    private static final long serialVersionUID = 0x050200L;

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * The number of remaining remote repliers. When this value
     * reaches zero, the request is completed.
     */
    public final int remaining;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new remote acknowledgment message.
     * @param remaining number of remaining remote repliers.
     *
     * @deprecated use {@link #RemoteAck(RemoteAck.Builder)} in
     * the future.
     */
    @Deprecated
    public RemoteAck(final int remaining)
    {
        super ();


        if (remaining < 0)
        {
            throw (
                new IllegalArgumentException("remaining < 0"));
        }

        this.remaining = remaining;
    } // end of RemoteAck(int)

    /**
     * Creates a new remote acknowledgment message from the
     * de-serialized information.
     * @param subject message subject.
     * @param timestamp message timestamp.
     * @param remaining number of remaining remote repliers.
     * @throws IllegalArgumentException
     * if {@code remaining} is &lt; zero.
     *
     * @deprecated use {@link #RemoteAck(RemoteAck.Builder)} in
     * the future.
     */
    @Deprecated
    public RemoteAck(final String subject,
                     final long timestamp,
                     final int remaining)
        throws IllegalArgumentException
    {
        super (subject, timestamp);

        if (remaining < 0)
        {
            throw (
                new IllegalArgumentException("remaining < 0"));
        }

        this.remaining = remaining;
    } // end of RemoteAck(String, long, int)

    private RemoteAck(final Builder builder)
    {
        super (builder);

        this.remaining = builder.mRemaining;
    } // end of RemoteAck(Builder)
    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Object Method Overrides.
    //

    @Override
    public boolean equals(final Object o)
    {
        boolean retcode = (this == o);

        if (retcode == false && o instanceof RemoteAck)
        {
            retcode = (super.equals(o) == true &&
                       remaining == ((RemoteAck) o).remaining);
        }

        return (retcode);
    } // end of equals(Object)

    @Override
    public int hashCode()
    {
        return ((super.hashCode() * 37) + remaining);
    } // end of hashCode()

    /**
     * Returns a human-readable text version of this message.
     * @return text version of this message.
     */
    @Override
    public String toString()
    {
        return (String.format("%s%n              remaining: %d",
                              super.toString(),
                              remaining));
    } // end of toString()

    //
    // end of Object Method Overrides.
    //-----------------------------------------------------------

    /**
     * Returns a new instance of the {@code RemoteAck} message
     * builder.
     * @return message builder instance.
     */
    public static Builder builder()
    {
        return (new Builder());
    } // end of builder()

//---------------------------------------------------------------
// Inner classes.
//

    /**
     * Class used to create {@link RemoteAck} messages. Used
     * by eBus to de-serialize an encoded message.
     */
    public static final class Builder
        extends ESystemMessage.Builder<RemoteAck, Builder>
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        private int mRemaining;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        private Builder()
        {
            super (RemoteAck.class);

            mRemaining = -1;
        } // end of Builder()

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Builder Method Overrides.
        //

        /**
         * Returns the newly instantiated {@code RemoteAck}
         * based on this builder configuration.
         * @return target message instance.
         */
        @Override
        protected RemoteAck buildImpl()
        {
            return (new RemoteAck(this));
        } // end of buildImpl()

        /**
         * Adds any and all configuration problems to the list.
         * @param problems list of misconfigurations.
         */
        @Override
        protected void validate(List<String> problems)
        {
            super.validate(problems);

            if (mRemaining < 0)
            {
                problems.add("remaining not set");
            }

            return;
        } // end of validate(List<>)

        //
        // end of Builder Method Overrides.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        /**
         * Sets the number of remaining repliers to the given
         * value.
         * @param n remaining repliers count.
         * @return {@code this RemoteAck} message builder.
         * @throws IllegalArgumentException
         * if {@code n} &lt; zero.
         */
        public Builder remaining(final int n)
        {
            if (n < 0)
            {
                throw (new IllegalArgumentException("n < zero"));
            }

            mRemaining = n;

            return (this);
        } // end of remaining(final int n)

        //
        // end of Set Methods.
        //-------------------------------------------------------
    } // end of class Builder
} // end of class RemoteAck
