//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2019. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.client.sysmessages;

import java.io.Serializable;
import java.time.Duration;
import java.util.List;
import net.sf.eBus.config.EConfigure.DiscardPolicy;

/**
 * This message is sent to request a connection pause with the
 * remote eBus application. A pause means that while the
 * connection is down, both sides will maintain the other side's
 * advertisements and queue up messages for future delivery. The
 * far-end eBus application will respond with a
 * {@link PauseReply} which specifies whether a pause is accepted
 * and, if yes, the pause time and maximum backlog size it
 * supports. These values may be the same as or less than this
 * message's suggested values; the response values will never be
 * greater than what is requested.
 * <p>
 * Contrast this message with a {@link LogoffMessage} which
 * tears down a connection completely.
 * </p>
 *
 * @see PauseReply
 * @see LogoffMessage
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public final class PauseRequest
    extends AbstractLogonMessage
    implements Serializable
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * Serialization version identifier.
     */
    private static final long serialVersionUID = 0x050200L;

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * Requesting a connection pause lasting for this duration.
     * The far-end may not accept this time and require a shorter
     * pause time
     */
    public final Duration pauseTime;

    /**
     * Requesting the far-end store up to this many un-delivered
     * messages before beginning to discard messages. Which
     * messages are to be discarded is based on the
     * {@link #discardPolicy}. Please note that is size is a
     * suggestion. The far-end response may specify smaller
     * backlog size.
     */
    public final int maximumBacklogSize;

    /**
     * Specifies how the far-end should discard messages if and
     * when {@link #maximumBacklogSize} is reached.
     */
    public final DiscardPolicy discardPolicy;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    private PauseRequest(final Builder builder)
    {
        super (builder);

        this.pauseTime = builder.mPauseTime;
        this.maximumBacklogSize = builder.mMaximumBacklogSize;
        this.discardPolicy = builder.mDiscardPolicy;
    } // end of PauseRequest(Builder)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Object Method Overrides.
    //

    @Override
    public String toString()
    {
        return (
            String.format(
                "%s%n             pause time: %s%n       max backlog size: %d%n        discard policy: %s",
                super.toString(),
                pauseTime,
                maximumBacklogSize,
                discardPolicy));
    } // end of toString()

    //
    // end of Object Method Overrides.
    //-----------------------------------------------------------

    /**
     * Returns a new instance of the {@code PauseRequest}
     * message builder.
     * @return message builder instance.
     */
    public static Builder builder()
    {
        return (new Builder());
    } // end of builder()

//---------------------------------------------------------------
// Inner classes.
//

    /**
     * Class used to create {@link PauseRequest} messages. Used
     * by eBus to de-serialize an encoded message.
     */
    public static final class Builder
        extends AbstractLogonMessage.Builder<PauseRequest, Builder>
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        /**
         * Requesting a connection pause lasting for this duration.
         * The far-end may not accept this time and require a shorter
         * pause time
         */
        private Duration mPauseTime;

        /**
         * Requesting the far-end store up to this many undelivered
         * messages before beginning to discard messages. Which
         * messages are to be discarded is based on the
         * {@link PauseRequest#discardPolicy}. Please note that
         * this size is a suggestion. The far-end response may
         * specify smaller backlog size.
         */
        private int mMaximumBacklogSize;

        /**
         * When the maximum message backlog size is exceeded,
         * discard messages as per this policy.
         */
        private DiscardPolicy mDiscardPolicy;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        private Builder()
        {
            super (PauseRequest.class);

            mMaximumBacklogSize = 0;
            mDiscardPolicy = DiscardPolicy.OLDEST_FIRST;
        } // end of Builder()

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Builder Method Overrides.
        //

        /**
         * Returns the newly instantiated {@code PauseRequest}
         * based on this builder configuration.
         * @return target message instance.
         */
        @Override
        protected PauseRequest buildImpl()
        {
            return (new PauseRequest(this));
        } // end of buildImpl()

        /**
         * Adds any and all configuration problems to the list.
         * @param problems list of mis-configurations.
         */
        @Override
        protected void validate(List<String> problems)
        {
            super.validate(problems);

            // Only pause time need be set. Maximum message
            // backlog size and discard policy have default
            // settings.
            if (mPauseTime == null)
            {
                problems.add("pause time not set");
            }

            return;
        } // end of validate(List<>)

        //
        // end of Builder Method Overrides.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        /**
         * Sets the pause time to the given value.
         * @param time pause time.
         * @return {@code this} pause request builder.
         * @throws NullPointerException
         * if {@code time} is {@code null}.
         */
        public Builder pauseTime(final Duration time)
        {
            if (time == null)
            {
                throw (new NullPointerException("time is null"));
            }

            mPauseTime = time;

            return (this);
        } // end of pauseTime(Duration)

        /**
         * Sets the maximum message backlog size to the given
         * value.
         * @param size maximum message backlog size.
         * @return {@code this} pause request builder.
         * @throws IllegalArgumentException
         * if {@code size} &lt; zero.
         */
        public Builder maximumBacklogSize(final int size)
        {
            if (size < 0)
            {
                throw (
                    new IllegalArgumentException("size < zero"));
            }

            mMaximumBacklogSize = size;

            return (this);
        } // end of maximumBacklogSize(int)

        /**
         * Sets the discard policy to the given value.
         * @param policy discard policy.
         * @return {@code this} pause request builder.
         * @throws NullPointerException
         * if {@code policy} is {@code null}.
         */
        public Builder discardPolicy(final DiscardPolicy policy)
        {
            if (policy == null)
            {
                throw (
                    new NullPointerException("policy is null"));
            }

            mDiscardPolicy = policy;

            return (this);
        } // end of discardPolicy(DiscardPolicy)

        //
        // end of Set Methods.
        //-------------------------------------------------------
    } // end of class Builder
} // end of class PauseRequest
