//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2019. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.client.sysmessages;

import java.io.Serializable;
import java.time.Duration;
import java.util.List;
import net.sf.eBus.messages.EReplyMessage.ReplyStatus;
import net.sf.eBus.messages.Optional;

/**
 * This message is sent is response to a {@link PauseRequest}.
 * This response specifies whether the pause request is accepted
 * or not. If accepted, then this response provides the allowed
 * pause duration and backlog size. These values will be &le; to
 * the requested values, never greater than. The requested
 * discard policy is always accepted.
 * <p>
 * If rejected, then {@link #replyReason} provided
 * human-readable text explaining the rejection. Please note that
 * {@link #replyStatus} will either be
 * {@link ReplyStatus#OK_FINAL} (accepted) or
 * {@link ReplyStatus#ERROR} (rejected); it will never be set to
 * {@link ReplyStatus#OK_CONTINUING}.
 * </p>
 *
 * @see PauseRequest
 * @see LogonReply
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public final class PauseReply
    extends AbstractLogonMessage
    implements Serializable
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * Serialization version identifier.
     */
    private static final long serialVersionUID = 0x050200L;

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * The current request handling status. If
     * {@link ReplyStatus#ERROR}, then {@link #replyReason} will
     * contain text explaining why the request was rejected.
     */
    public final ReplyStatus replyStatus;

    /**
     * The reason for a {@link ReplyStatus#ERROR} reply status.
     * May be {@code null} or empty.
     */
    @Optional
    public final String replyReason;

    /**
     * Requesting a connection pause lasting for this duration.
     * The far-end may not accept this time and require a shorter
     * pause time
     */
    public final Duration pauseTime;

    /**
     * Requesting the far-end store up to this many undelivered
     * messages before beginning to discard messages. Which
     * messages are to be discarded is based on the
     * {@link PauseRequest#discardPolicy}. Please note that this
     * size is a suggestion. The far-end response may specify
     * smaller backlog size.
     */
    public final int maximumBacklogSize;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a pause reply message from the configured builder.
     * @param builder pause reply message builder.
     */
    private PauseReply(final Builder builder)
    {
        super (builder);

        this.replyStatus = builder.mReplyStatus;
        this.replyReason = builder.mReplyReason;
        this.pauseTime = builder.mPauseTime;
        this.maximumBacklogSize = builder.mMaximumBacklogSize;
    } // end of PauseReply(Builder)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Object Method Overrides.
    //

    @Override
    public String toString()
    {
        return (
            String.format(
                "%s%n                 status: %s%n                reason: %s%n             pause time: %s%n       max backlog size: %d",
                super.toString(),
                replyStatus,
                replyReason,
                pauseTime,
                maximumBacklogSize));
    } // end of toString()

    //
    // end of Object Method Overrides.
    //-----------------------------------------------------------

    /**
     * Returns a new instance of the {@code PauseReply} message
     * builder.
     * @return message builder instance.
     */
    public static Builder builder()
    {
        return (new Builder());
    } // end of builder()

//---------------------------------------------------------------
// Inner classes.
//

    /**
     * Class used to create {@link PauseReply} messages. Used
     * by eBus to de-serialize an encoded message.
     */
    public static final class Builder
        extends AbstractLogonMessage.Builder<PauseReply, Builder>
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        /**
         * The current request handling status. If
         * {@link ReplyStatus#ERROR}, then {@link #replyReason} will
         * contain text explaining why the request was rejected.
         */
        private ReplyStatus mReplyStatus;

        /**
         * The reason for a {@link ReplyStatus#ERROR} reply status.
         * May be {@code null} or empty.
         */
        private String mReplyReason;

        /**
         * Requesting a connection pause lasting for this duration.
         * The far-end may not accept this time and require a shorter
         * pause time
         */
        private Duration mPauseTime;

        /**
         * Requesting the far-end store up to this many undelivered
         * messages before beginning to discard messages. Which
         * messages are to be discarded is based on the
         * {@link PauseRequest#discardPolicy}. Please note that
         * this size is a suggestion. The far-end response may
         * specify smaller backlog size.
         */
        private int mMaximumBacklogSize;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        private Builder()
        {
            super (PauseReply.class);

            mMaximumBacklogSize = 0;
        } // end of Builder()

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Builder Method Overrides.
        //

        /**
         * Returns the newly instantiated {@code PauseReply}
         * based on this builder configuration.
         * @return target message instance.
         */
        @Override
        protected PauseReply buildImpl()
        {
            return (new PauseReply(this));
        } // end of buildImpl()

        /**
         * Adds any and all configuration problems to the list.
         * @param problems list of mis-configurations.
         */
        @Override
        protected void validate(List<String> problems)
        {
            super.validate(problems);

            // Reply status and pause time must be set.
            // Reply reason and maximum backlog size have
            // default settings.
            if (mReplyStatus == null)
            {
                problems.add("reply status not set");
            }

            if (mPauseTime == null)
            {
                problems.add("pause time not set");
            }

            return;
        } // end of validate(List<>)

        //
        // end of Object Method Overrides.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        /**
         * Sets the reply status to the given value.
         * @param status reply status.
         * @return {@code this} pause reply builder.
         * @throws NullPointerException
         * if {@code status} is {@code null}.
         */
        public Builder replyStatus(final ReplyStatus status)
        {
            if (status == null)
            {
                throw (
                    new NullPointerException("status is null"));
            }

            mReplyStatus = status;

            return (this);
        } // end of replyStatus(ReplyStatus)

        /**
         * Sets the reply reason to the given value. May be
         * {@code null} or an empty string.
         * @param reason reply reason.
         * @return {@code this} pause reply builder.
         */
        public Builder replyReason(final String reason)
        {
            mReplyReason = reason;

            return (this);
        } // end of replyReason(String)

        /**
         * Sets the pause time to the given value.
         * @param time pause time.
         * @return {@code this} pause reply builder.
         * @throws NullPointerException
         * if {@code time} is {@code null}.
         */
        public Builder pauseTime(final Duration time)
        {
            if (time == null)
            {
                throw (new NullPointerException("time is null"));
            }

            mPauseTime = time;

            return (this);
        } // end of pauseTime(Duration)

        /**
         * Sets the maximum message backlog size to the given
         * value.
         * @param size maximum message backlog size.
         * @return {@code this} pause reply builder.
         * @throws IllegalArgumentException
         * if {@code size} &lt; zero.
         */
        public Builder maximumBacklogSize(final int size)
        {
            if (size < 0)
            {
                throw (
                    new IllegalArgumentException("size < zero"));
            }

            mMaximumBacklogSize = size;

            return (this);
        } // end of maximumBacklogSize(int)

        //
        // end of Set Methods.
        //-------------------------------------------------------
    } // end of class Builder
} // end of class PauseReply
