//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2012, 2013, 2016, 2019. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.client.sysmessages;

import java.io.Serializable;
import java.util.List;
import net.sf.eBus.messages.EReplyMessage.ReplyStatus;
import net.sf.eBus.messages.Optional;

/**
 * This message is sent in reply to a logon message. The subject
 * contains the eBus identifier received in the
 * {@link LogonMessage}.
 *
 * @see AbstractLogonMessage
 * @see LogonMessage
 * @see LogoffMessage
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public final class LogonReply
    extends AbstractLogonMessage
    implements Serializable
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * Serialization version identifier.
     */
    private static final long serialVersionUID = 0x050200L;

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * Either the logon attempt succeeded or failed.
     */
    public final ReplyStatus logonStatus;

    /**
     * Explanation for a rejected logon request. May be
     * {@code null} or empty.
     */
    @Optional
    public final String reason;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    private LogonReply(final Builder builder)
    {
        super (builder);

        this.logonStatus = builder.mLogonStatus;
        this.reason = builder.mReason;
    } // end of LogonReply(Builder)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Object Method Overrides.
    //

    /**
     * Returns {@code true} if {@code o} is a
     * non-{@code null LogonReply} instance with reply status and
     * reason equal to {@code this LogonReply} instance and
     * {@code false} otherwise.
     * @param o comparison object.
     * @return {@code true} if the message fields are equal
     * and {@code false} otherwise.
     */
    @Override
    public boolean equals(final Object o)
    {
        boolean retcode = (this == o);

        if (retcode == false && o instanceof LogonReply)
        {
            final LogonReply lr = (LogonReply) o;

            retcode = (super.equals(o) == true &&
                       logonStatus == lr.logonStatus &&
                       (reason == null ?
                            lr.reason == null :
                            reason.equals(lr.reason) == true));
        }

        return (retcode);
    } // endof equals(Object)

    /**
     * Returns the feed status message hash code.
     * @return the feed status message hash code.
     */
    @Override
    public int hashCode()
    {
        return (
            (((super.hashCode() * 37) +
              logonStatus.ordinal()) * 37) +
            (reason == null ? 0 : reason.hashCode()));
    } // end of hashCode()

    /**
     * Returns a human-readable text version of this message.
     * @return text version of this message.
     */
    @Override
    public String toString()
    {
        return (
            String.format(
                "%s%n           logon status: %s%n                 reason: %s",
                super.toString(),
                logonStatus,
                (reason == null ? "(none)" : reason)));
    } // end of toString()

    //
    // end of Object Method Overrides.
    //-----------------------------------------------------------

    /**
     * Returns a new instance of the {@code LogonReply} builder.
     * @return message builder instance.
     */
    public static Builder builder()
    {
        return (new Builder());
    } // end of builder()

//---------------------------------------------------------------
// Inner classes.
//

    /**
     * Class used to create {@link LogonReply} instances.
     * Used by eBus to de-serialize an encoded message.
     */
    public static final class Builder
        extends AbstractLogonMessage.Builder<LogonReply, Builder>
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        /**
         * Required reply status field specifying whether the
         * logon request was accepted or rejected.
         */
        private ReplyStatus mLogonStatus;

        /**
         * Optional field explaining why logon request is
         * rejected.
         */
        private String mReason;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        private Builder()
        {
            super (LogonReply.class);
        } // end of Builder()

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Builder Method Overrides.
        //

        /**
         * Returns the newly instantiated
         * {@code LogonReplyMessage} based on this builder
         * configuration.
         * @return target message instance.
         */
        @Override
        protected LogonReply buildImpl()
        {
            return (new LogonReply(this));
        } // end of buildImpl()

        /**
         * Checks if logon status is set. If not then failure is
         * added to the problems list.
         * @param problems list of message builder problems.
         */
        @Override
        protected void validate(final List<String> problems)
        {
            super.validate(problems);

            if (mLogonStatus == null)
            {
                problems.add("logon status not set");
            }

            return;
        } // end of validate(List<>)

        //
        // end of Builder Method Overrides.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        /**
         * Sets the logon status to the given value.
         * @param status logon status.
         * @return {@code this} logon reply builder.
         * @throws NullPointerException
         * if {@code status} is {@code null}.
         */
        public Builder logonStatus(final ReplyStatus status)
        {
            if (status == null)
            {
                throw (
                    new NullPointerException("status is null"));
            }

            mLogonStatus = status;

            return (this);
        } // end of logonStatus(final ReplyStatus status)

        /**
         * Sets the logon status reason.
         * @param t reason for logon status. May be a
         * {@code null} or empty string.
         * @return {@code this} logon reply builder.
         */
        public Builder reason(final String t)
        {
            mReason = t;

            return (this);
        } // end of reason(final String t)

        //
        // end of Set Methods.
        //-------------------------------------------------------
    } // end of class Builder
} // end of class LogonReply
