//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2016, 2019. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.client.sysmessages;

import com.google.common.base.Strings;
import java.io.Serializable;
import java.util.List;
import net.sf.eBus.messages.EMessageKey;
import net.sf.eBus.messages.ESystemMessage;

/**
 * This system message is used to exchange
 * message key-to-identifier mappings.
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public final class KeyMessage
    extends ESystemMessage
    implements Serializable
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * Serialization version identifier.
     */
    private static final long serialVersionUID = 0x050200L;

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * The sender's message key identifier. The receiver may
     * assign its own unique identifier for the key.
     */
    public final int keyId;

    /**
     * The message key class name.
     */
    public final String keyClass;

    /**
     * The message key subject.
     */
    public final String keySubject;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new key message from the configured message
     * builder.
     * @param builder message builder.
     */
    private KeyMessage(final Builder builder)
    {
        super (builder);

        this.keyId = builder.mKeyId;
        this.keyClass = builder.mKeyClass;
        this.keySubject = builder.mKeySubject;
    } // end of KeyMessage(Builder)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Object Method Overrides.
    //

    @Override
    public boolean equals(final Object o)
    {
        boolean retcode = (this == o);

        if (retcode == false &&
            o instanceof KeyMessage)
        {
            retcode = (super.equals(o) == true &&
                       keyId == ((KeyMessage) o).keyId);
        }

        return (retcode);
    } // end of equals(Object)

    @Override
    public int hashCode()
    {
        return ((super.hashCode() * 37) + keyId);
    } // end of hashCode()

    @Override
    public String toString()
    {
        return (
            String.format("%s%n                key ID: %d%n             class name: %s%n                subject: %s",
                super.toString(),
                keyId,
                keyClass,
                keySubject));
    } // end of toString()

    //
    // end of Object Method Overrides.
    //-----------------------------------------------------------

    /**
     * Returns a new instance of the {@code KeyMessage} builder.
     * @return message builder instance.
     */
    public static Builder builder()
    {
        return (new Builder());
    } // end of builder()

//---------------------------------------------------------------
// Inner classes.
//

    /**
     * Class used to create {@link KeyMessage} instances. Used by
     * eBus to de-serialize an encoded message.
     */
    public static final class Builder
        extends ESystemMessage.Builder<KeyMessage, Builder>
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        private int mKeyId;
        private String mKeyClass;
        private String mKeySubject;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        private Builder()
        {
            super (KeyMessage.class);
        } // end of Builder()

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Builder Method Overrides.
        //

        /**
         * Returns the newly instantiated {@code KeyMessage}
         * based on this builder configuration.
         * @return target message instance.
         */
        @Override
        protected KeyMessage buildImpl()
        {
            throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
        }

        /**
         * Checks if the key class and subject are configured.
         * If not, then appends those problems to the list.
         * @param problems list of message configuration
         * problems.
         */
        @Override
        protected void validate(List<String> problems)
        {
            super.validate(problems);

            if (mKeyClass == null)
            {
                problems.add("key class not set");
            }

            if (mKeySubject == null)
            {
                problems.add("key subject not set");
            }

            return;
        } // end of validate(List<>)

        //
        // end of Builder Method Overrides.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        /**
         * Sets the message key identifier to the given value.
         * @param id message key identifier.
         * @return {@code this} key message builder.
         */
        public Builder keyId(final int id)
        {
            mKeyId = id;

            return (this);
        } // end of keyId(int)

        /**
         * Sets the key message class name to the given value.
         * @param mc key message class name.
         * @return {@code this} key message builder.
         * @throws IllegalArgumentException
         * if {@code mc} is {@code null} or empty.
         */
        public Builder keyClass(final String mc)
        {
            if (Strings.isNullOrEmpty(mc))
            {
                throw (
                    new IllegalArgumentException(
                        "mc is null or empty"));
            }

            mKeyClass = mc;

            return (this);
        } // end of keyClass(String)

        /**
         * Sets the key message subject to the given value.
         * @param subject message subject.
         * @return {@code this} key message builder.
         * @throws IllegalArgumentException
         * if {@code subject} is {@code null} or empty.
         */
        public Builder keySubject(final String subject)
        {
            if (Strings.isNullOrEmpty(subject))
            {
                throw (
                    new IllegalArgumentException(
                        "subject is null or empty"));
            }

            mKeySubject = subject;

            return (this);
        } // end of keySubject(String)

        /**
         * Sets the message class name and subject based on the
         * given message key.
         * @param key take class name and subject from this
         * message key.
         * @return {@code this} key message builder.
         * @throws NullPointerException
         * if {@code key} is {@code null}.
         */
        public Builder key(final EMessageKey key)
        {
            if (key == null)
            {
                throw (new NullPointerException("key is null"));
            }

            mKeyClass = key.className();
            mKeySubject = key.subject();

            return (this);
        } // end of key(EMessageKey)

        //
        // end of Set Methods.
        //-------------------------------------------------------
    } // end of class Builder
} // end of class KeyMessage
