//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2012, 2013, 2016, 2019. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.client.sysmessages;

import java.io.Serializable;
import net.sf.eBus.client.EFeedState;
import net.sf.eBus.messages.ESystemMessage;

/**
 * Forwards a notification feed status to a remote eBus
 * application so it can be reported to the remote subscribers.
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public final class FeedStatusMessage
    extends ESystemMessage
    implements Serializable
{

//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * Serialization version identifier.
     */
    private static final long serialVersionUID = 0x050200L;

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * The updated feed state.
     */
    public final EFeedState feedState;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new feed status message from the configured
     * message builder.
     * @param builder message builder.
     */
    private FeedStatusMessage(final Builder builder)
    {
        super (builder);

        this.feedState = builder.mFeedState;
    } // end of FeedStatusMessage(Builder)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Object Method Overrides.
    //

    /**
     * Returns {@code true} if {@code o} is a
     * non-{@code null FeedStatusMessage} instance with feed
     * status and reason equal to {@code this FeedStatusMessage}
     * instance and {@code false} otherwise.
     * @param o comparison object.
     * @return {@code true} if the message fields are equal
     * and {@code false} otherwise.
     */
    @Override
    public boolean equals(final Object o)
    {
        boolean retcode = (this == o);

        if (retcode == false && o instanceof FeedStatusMessage)
        {
            final FeedStatusMessage fs = (FeedStatusMessage) o;

            retcode = (super.equals(o) == true &&
                       feedState == fs.feedState);
        }

        return (retcode);
    } // end of equals(Object)

    /**
     * Returns the feed status message hash code.
     * @return the feed status message hash code.
     */
    @Override
    public int hashCode()
    {
        return (((super.hashCode() * 37) + feedState.ordinal()));
    } // end of hashCode()

    /**
     * Returns a human-readable text version of this message.
     * @return text version of this message.
     */
    @Override
    public String toString()
    {
        return (
            String.format(
                "%s%n             feed state: %s",
                super.toString(),
                feedState));
    } // end of toString()

    //
    // end of Object Method Overrides.
    //-----------------------------------------------------------

    /**
     * Returns a new instance of the {@code FeedStatusMessage}
     * builder.
     * @return message builder instance.
     */
    public static Builder builder()
    {
        return (new Builder());
    } // end of builder()

//---------------------------------------------------------------
// Inner classes.
//

    /**
     * Class used to create {@link FeedStatusMessage} instances.
     * Used by eBus to de-serialize an encoded message.
     */
    public static final class Builder
        extends ESystemMessage.Builder<FeedStatusMessage, Builder>
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        private EFeedState mFeedState;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        private Builder()
        {
            super (FeedStatusMessage.class);
        } // end of Builder()

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Builder Method Overrides.
        //

        @Override
        protected FeedStatusMessage buildImpl()
        {
            return (new FeedStatusMessage(this));
        } // end of buildImpl()

        //
        // end of Builder Method Overrides.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        /**
         * Sets the feed state to the given value.
         * @param fs advertisement feed state. May be
         * {@code null}.
         * @return {@code this} builder.
         */
        public Builder feedState(final EFeedState fs)
        {
            mFeedState = fs;

            return (this);
        } // end of feedState(final EFeedState fs)

        //
        // end of Set Methods.
        //-------------------------------------------------------
    } // end of class Builder
} // end of class FeedStatus
