//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2011, 2013, 2015, 2016, 2019. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.client.sysmessages;

import java.io.Serializable;
import java.util.List;
import net.sf.eBus.client.EFeedState;
import net.sf.eBus.messages.EMessage.MessageType;

/**
 * Reports either a notification publisher or a request replier
 * advertisement to a remote eBus application and whether the
 * advertisement should be put into place or retracted.
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public final class AdMessage
    extends AbstractKeyMessage
    implements Serializable
{
//---------------------------------------------------------------
// Enums.
//

    /**
     * This enum specifies whether an advertisement should be
     * added or removed.
     */
    public enum AdStatus
    {
        /**
         * Put the advertisement in place.
         */
        ADD,

        /**
         * Retract the advertisement.
         */
        REMOVE
    } // end of enum AdStatus

//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * Serialization version identifier.
     */
    private static final long serialVersionUID = 0x050200L;

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * Add or retract this advertisement.
     */
    public final AdStatus adStatus;

    /**
     * The advertised message is for either a notification or a
     * request.
     */
    public final MessageType adMessageType;

    /**
     * The advertised feed's current state.
     */
    public final EFeedState feedState;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new advertisement message from the configured
     * message builder.
     * @param builder message builder.
     */
    private AdMessage(final Builder builder)
    {
        super (builder);

        this.adStatus = builder.mAdStatus;
        this.adMessageType = builder.mMessageType;
        this.feedState = builder.mFeedState;
    } // end of method(builder)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Object Method Overrides.
    //

    /**
     * Returns {@code true} if {@code o} is a
     * non-{@code null AdMessage} instance with ad status,
     * message class, and message type equal to
     * {@code this AdMessage} instance and {@code false}
     * otherwise.
     * @param o comparison object.
     * @return {@code true} if the message fields are equal
     * and {@code false} otherwise.
     */
    @Override
    public boolean equals(final Object o)
    {
        boolean retcode = (this == o);

        if (retcode == false && o instanceof AdMessage)
        {
            final AdMessage adMsg = (AdMessage) o;

            retcode = (super.equals(o) == true &&
                       adStatus == adMsg.adStatus &&
                       adMessageType == adMsg.adMessageType &&
                       feedState == adMsg.feedState);
        }

        return (retcode);
    } // end of equals(Object)

    /**
     * Returns the advertisement message hash code.
     * @return the advertisement message hash code.
     */
    @Override
    public int hashCode()
    {
        return (
            (((((super.hashCode() * 37) +
                adStatus.ordinal()) * 37) +
              adMessageType.ordinal()) * 37) +
            feedState.ordinal());
    } // end of hashCode()

    /**
     * Returns a human-readable text version of this message.
     * @return text version of this message.
     */
    @Override
    public String toString()
    {
        return (
            String.format(
                "%s%n       advertise status: %s%n           message type: %s%n             feed state: %s",
                super.toString(),
                adStatus,
                adMessageType,
                feedState));
    } // end of toString()

    //
    // end of Object Method Overrides.
    //-----------------------------------------------------------

    /**
     * Returns a new instance of the {@code AdMessage} builder.
     * @return message builder instance.
     */
    public static Builder builder()
    {
        return (new Builder());
    } // end of builder()

//---------------------------------------------------------------
// Inner classes.
//

    /**
     * Class used to create {@link AdMessage} instances. Used by
     * eBus to de-serialize an encoded message.
     */
    public static final class Builder
        extends AbstractKeyMessage.Builder<AdMessage, Builder>
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        private AdStatus mAdStatus;
        private MessageType mMessageType;
        private EFeedState mFeedState;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        private Builder()
        {
            super (AdMessage.class);
        } // end of Builder()

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Builder Method Overrides.
        //

        /**
         * Returns the newly instantiated {@code AdMessage} based
         * on this builder configuration.
         * @return target message instance.
         */
        @Override
        protected AdMessage buildImpl()
        {
            return (new AdMessage(this));
        } // end of buildImpl()

        /**
         * Checks if the ad status and message type are
         * configured. If not, then appends those problems to the
         * list. Note that feed state does not need to be
         * configured.
         * @param problems list of message configuration
         * problems.
         */
        @Override
        protected void validate(final List<String> problems)
        {
            super.validate(problems);

            if (mAdStatus == null)
            {
                problems.add("ad status not set");
            }

            if (mMessageType == null)
            {
                problems.add("message type not set");
            }

            return;
        } // end of validate(List<>)

        //
        // end of Builder Method Overrides.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        /**
         * Sets the advertisement status to the given value.
         * @param status advertisement status.
         * @return {@code this} builder.
         * @throws NullPointerException
         * if {@code status} is {@code null}.
         */
        public Builder adStatus(final AdStatus status)
        {
            if (status == null)
            {
                throw (
                    new NullPointerException("status is null"));
            }

            mAdStatus = status;

            return (this);
        } // end of adStatus(AdStatus)

        /**
         * Sets the message type to the given value.
         * @param mt message type.
         * @return {@code this} builder.
         * @throws NullPointerException
         * if {@code mt} is {@code null}.
         */
        public Builder adMessageType(final MessageType mt)
        {
            if (mt == null)
            {
                throw (new NullPointerException("mt is null"));
            }

            mMessageType = mt;

            return (this);
        } // end of adMessageType(MessageType)

        /**
         * Sets the advertisement feed state to the given value.
         * @param fs advertisement feed state. May be
         * {@code null}.
         * @return {@code this} builder.
         */
        public Builder feedState(final EFeedState fs)
        {
            mFeedState = fs;

            return (this);
        } // end of feedState(EFeedState)

        //
        // end of Set Methods.
        //-------------------------------------------------------
    } // end of class Builder
} // end of class AdMessage
