//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2013, 2016, 2019. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.client.sysmessages;

import com.google.common.base.Strings;
import java.io.Serializable;
import java.util.List;
import net.sf.eBus.messages.EMessageObject;
import net.sf.eBus.messages.ESystemMessage;

/**
 * Base class for the eBus logon messages {@link LogonMessage},
 * {@link LogonReply}, and {@link LogoffMessage}. Contains the
 * unique eBus identifier. For a Java application, this is the
 * JVM identifier and for a .Net application, the process
 * identifier.
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public abstract class AbstractLogonMessage
    extends ESystemMessage
    implements Serializable
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * Serialization version identifier.
     */
    private static final long serialVersionUID = 0x050200L;

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * The eBus identifier.
     */
    public final String eid;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new abstract logon message from the configured
     * message builder.
     * @param builder message builder.
     */
    protected AbstractLogonMessage(final Builder<?, ?> builder)
    {
        super (builder);

        eid = builder.mEid;
    } // end of AbstractLogonMessage(Builder)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Object Method Overrides.
    //

    /**
     * Returns {@code true} if {@code o} is a
     * non-{@code null AbstractLogonMessage} instance with an
     * eBus identifier equal to
     * {@code this AbstractLogonMessage} instance and
     * {@code false} otherwise.
     * @param o comparison object.
     * @return {@code true} if the message fields are equal
     * and {@code false} otherwise.
     */
    @Override
    public boolean equals(final Object o)
    {
        boolean retcode = (this == o);

        if (retcode == false &&
            o instanceof AbstractLogonMessage)
        {
            final AbstractLogonMessage alm =
                (AbstractLogonMessage) o;

            retcode = (super.equals(o) == true &&
                       eid.equals(alm.eid) == true);
        }

        return (retcode);
    } // end of equals(Object)

    /**
     * Returns the feed status message hash code.
     * @return the feed status message hash code.
     */
    @Override
    public int hashCode()
    {
        return ((super.hashCode() * 37) + eid.hashCode());
    } // end of hashCode()

    /**
     * Returns a human-readable text version of this message.
     * @return text version of this message.
     */
    @Override
    public String toString()
    {
        return (
            String.format(
                "%s%n                eBus ID: %s",
                super.toString(),
                eid));
    } // end of toString()

    //
    // end of Object Method Overrides.
    //-----------------------------------------------------------

//---------------------------------------------------------------
// Inner classes.
//

    /**
     * Base class for all {@link AbstractLogonMessage} builders.
     * Provides eBus identifier validation.
     *
     * @param <M> builds this target message class.
     * @param <B> message builder subclass. Needed to return the
     * correct builder type when setting fields. If this were not
     * the case, field chaining would not work.
     */
    public static abstract class Builder<M extends AbstractLogonMessage,
                                         B extends Builder<M, ?>>
        extends ESystemMessage.Builder<M, B>
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        /**
         * eBus identifier.
         */
        private String mEid;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        protected Builder(final Class<? extends EMessageObject> targetClass)
        {
            super (targetClass);
        } // end of Builder(Class)

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Builder Method Overrides.
        //

        /**
         * Checks if the eBus identifier is configured. If not,
         * then appends those problems to the list.
         * @param problems list of message configuration
         * problems.
         */
        @Override
        protected void validate(final List<String> problems)
        {
            super.validate(problems);

            if (mEid == null)
            {
                problems.add("eid not set");
            }

            return;
        } // end of validate(List<>)

        //
        // end of Builder Method Overrides.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        /**
         * Sets the eBus identifier to the given value.
         * @param eid eBus identifier.
         * @return {@code this} message builder.
         * @throws IllegalArgumentException
         * if {@code eid} is {@code null} or empty.
         */
        @SuppressWarnings ("unchecked")
        public B eid(final String eid)
        {
            if (Strings.isNullOrEmpty(eid))
            {
                throw (
                    new IllegalArgumentException(
                        "eid is null or empty"));
            }

            mEid = eid;

            return ((B) this);
        } // end of eid(String)

        //
        // end of Set Methods.
        //-------------------------------------------------------
    } // end of class Builder<M, B>
} // end of class AbstractLogonMessage
