//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2015, 2016, 2019. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.client.sysmessages;

import com.google.common.base.Strings;
import java.io.Serializable;
import java.util.List;
import net.sf.eBus.messages.EMessage;
import net.sf.eBus.messages.EMessageKey;
import net.sf.eBus.messages.EMessageObject;
import net.sf.eBus.messages.ESystemMessage;
import net.sf.eBus.messages.UnknownMessageException;

/**
 * Base class for all system messages which contain a message
 * class name and message subject. The reason for sending a
 * message key as a separate message class name and subject is
 * due to the receiving eBus application failing to de-serialize
 * the message key because {@code Class.forName} throwing
 * {@code ClassNotFoundException}. It is entirely reasonable for
 * connected eBus applications not to use exactly the same
 * message set. Therefore, if an eBus application receives a
 * message class name for a class it doesn't know about, that
 * inbound message will successfully deserialize but is ignored.
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public abstract class AbstractKeyMessage
    extends ESystemMessage
    implements Serializable
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * Serialization version identifier.
     */
    private static final long serialVersionUID = 0x050200L;

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * The message key class name. The class name is transmitted
     * rather than the {@code Class} instance because the far-end
     * deserialization will fail if the {@code Class} is
     * unknown.
     */
    public final String messageClass;

    /**
     * The message key subject.
     */
    public final String messageSubject;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new abstract key message from the configured
     * message builder.
     * @param builder message builder.
     */
    protected AbstractKeyMessage(final Builder<?, ?> builder)
    {
        super (builder);

        this.messageClass = builder.mMessageClass;
        this.messageSubject = builder.mMessageSubject;
    } // end of AbstractKeyMessage(Builder)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Object Method Overrides.
    //

    /**
     * Returns {@code true} if {@code o} is a
     * non-{@code null AdMessage} instance with ad status,
     * message class, and message type equal to
     * {@code this AdMessage} instance and {@code false}
     * otherwise.
     * @param o comparison object.
     * @return {@code true} if the message fields are equal
     * and {@code false} otherwise.
     */
    @Override
    public boolean equals(final Object o)
    {
        boolean retcode = (this == o);

        if (retcode == false && o instanceof AbstractKeyMessage)
        {
            final AbstractKeyMessage keyMsg =
                (AbstractKeyMessage) o;

            retcode =
                (super.equals(o) == true &&
                 messageClass.equals(
                     keyMsg.messageClass) == true &&
                 messageSubject.equals(
                     keyMsg.messageSubject) == true);
        }

        return (retcode);
    } // end of equals(Object)

    /**
     * Returns the advertisement message hash code.
     * @return the advertisement message hash code.
     */
    @Override
    public int hashCode()
    {
        return (
            (((super.hashCode() * 37) +
              messageClass.hashCode()) * 37) +
            messageSubject.hashCode());
    } // end of hashCode()

    /**
     * Returns a human-readable text version of this message.
     * @return text version of this message.
     */
    @Override
    public String toString()
    {
        return (
            String.format(
                "%s%n            message key: %s/%s",
                super.toString(),
                messageClass,
                messageSubject));
    } // end of toString()

    //
    // end of Object Method Overrides.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get Methods.
    //

    /**
     * Returns the message key based on the message class name
     * and subject.
     * @return the message key.
     * @throws UnknownMessageException
     * if {@link #messageClass} is an unknown class name.
     */
    @SuppressWarnings ("unchecked")
    public final EMessageKey messageKey()
        throws UnknownMessageException
    {
        final Class<? extends EMessage> mc;

        try
        {
            mc =
                (Class<? extends EMessage>)
                    Class.forName(messageClass);
        }
        catch (ClassNotFoundException classex)
        {
            throw (new UnknownMessageException(messageClass));
        }

        return (new EMessageKey(mc, messageSubject));
    } // end of messageKey()

    //
    // end of Get Methods.
    //-----------------------------------------------------------

//---------------------------------------------------------------
// Member methods.
//

    /**
     * Base class for all {@link AbstractKeyMessage} builders.
     * Provides message class and subject validation.
     *
     * @param <M> builds this target message class.
     * @param <B> message builder subclass. Needed to return the
     * correct builder type when setting fields. If this were not
     * the case, field chaining would not work.
     */
    public static abstract class Builder<M extends AbstractKeyMessage,
                                         B extends Builder<M, ?>>
        extends ESystemMessage.Builder<M, B>
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        /**
         * Configured message class.
         */
        private String mMessageClass;

        /**
         * Configured message subject.
         */
        private String mMessageSubject;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        protected Builder(final Class<? extends EMessageObject> targetClass)
        {
            super (targetClass);
        } // end of Builder(Class)

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Builder Method Overrides.
        //

        /**
         * Checks if the message class and subject are
         * configured. If not, then appends those problems to the
         * list.
         * @param problems list of message configuration
         * problems.
         */
        @Override
        protected void validate(final List<String> problems)
        {
            super.validate(problems);

            if (mMessageClass == null)
            {
                problems.add("message class not set");
            }

            if (mMessageSubject == null)
            {
                problems.add("message subject not set");
            }

            return;
        } // end of validate(List<>)

        //
        // end of Builder Method Overrides.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        /**
         * Sets the message class name to the given value.
         * @param mc message class name.
         * @return {@code this} message builder.
         * @throws IllegalArgumentException
         * if {@code mc} is {@code null} or empty.
         */
        @SuppressWarnings ("unchecked")
        public B messageClass(final String mc)
        {
            if (Strings.isNullOrEmpty(mc))
            {
                throw (
                    new IllegalArgumentException(
                        "mc is null or empty"));
            }

            mMessageClass = mc;

            return ((B) this);
        } // end of messageClass(String)

        /**
         * Sets the message subject to the given value.
         * @param subject message subject.
         * @return {@code this} message builder.
         * @throws IllegalArgumentException
         * if {@code subject} is {@code null} or empty.
         */
        @SuppressWarnings ("unchecked")
        public B messageSubject(final String subject)
        {
            if (Strings.isNullOrEmpty(subject))
            {
                throw (
                    new IllegalArgumentException(
                        "subject is null or empty"));
            }

            mMessageSubject = subject;

            return ((B) this);
        } // end of messageSubject(final String subject)

        /**
         * Sets both the message class name and subject based on
         * the given message key.
         * @param key message key containing message class and
         * subject.
         * @return {@code this} message builder.
         * @throws NullPointerException
         * if {@code key} is {@code null}.
         */
        @SuppressWarnings ("unchecked")
        public B messageKey(final EMessageKey key)
        {
            if (key == null)
            {
                throw (new NullPointerException("key is null"));
            }

            mMessageClass = key.className();
            mMessageSubject = key.subject();

            return ((B) this);
        } // end of messageKey(final EMessageKey key)

        //
        // end of Set Methods.
        //-------------------------------------------------------
    } // end of class Builder<M, B>
} // end of class AbstractKeyMessage
