//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2015. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.client;

/**
 * Lists the thread types available to eBus users when
 * configuring eBus. A dispatcher thread either:
 * <ul>
 *   <li>
 *     blocks while waiting for the desire event to occur,
 *   </li>
 *   <li>
 *     spins, repeatedly check for the event to occur, or
 *   </li>
 *   <li>
 *     spins a specified number of times on the event check and,
 *     if the event does not occur when the spin limit is
 *     reached, then
 *     {@link java.util.concurrent.locks.LockSupport#parkNanos(long) parks}
 *     for the nanosecond time.
 *   </li>
 *   <li>
 *     spins a specified number of times on the event check and,
 *     if the event does not occur when the spin limit is
 *     reached, then
 *     {@link java.util.concurrent.locks.LockSupport#park() parks}
 *     for an indefinite time.
 *   </li>
 * </ul>
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public enum ThreadType
{
    /**
     * The thread blocks while waiting for the desired event
     * to occur. The thread continues if either the event
     * happened or an interrupt was caught. It is likely that a
     * thread using this type will be moved off core when
     * blocked.
     */
    BLOCKING ("blocking"),

    /**
     * The thread repeatedly checks if the event has arrived
     * using a non-blocking call without pausing in between
     * checks. This thread type effectively monopolizes a core and
     * keeps the thread on core.
     */
    SPINNING ("spinning"),

    /**
     * This thread repeatedly checks for the event using a
     * non-blocking call but only for a fixed number of times
     * (the spin limit). When the spin limit is reached, the
     * thread
     * {@link java.util.concurrent.locks.LockSupport#parkNanos(long) parks}
     * for a specified nanosecond time limit. Upon waking up,
     * the threads resets the spin count and starts the repeated,
     * non-blocking checks again. This continues until an event
     * is detected.
     * <p>
     * This type is a compromise between blocking and spinning
     * by aggressively spinning until the event occurs but not
     * entirely monopolizing a core. By parking, the thread may
     * be moved off core but not as likely as in a blocking
     * thread.
     * </p>
     */
    SPINPARK ("spin+park"),

    /**
     * This thread repeatedly check for the event using a
     * non-blocking call but only for a fixed number of times
     * (the spin limit). When the spin limit is reached, the
     * thread
     * {@link java.util.concurrent.locks.LockSupport#park() yields}
     * the core to another thread. When the thread re-gains the
     * core, the the spin count is reset and the repeated,
     * non-blocking checks start again. This continues until an
     * event is detected.
     * <p>
     * This type is a compromise between blocking and spinning
     * by aggressively spinning until an event occurs but not
     * entirely monopolizing a core. By yielding, the thread
     * may be moved off core but not as likely as in a blocking
     * thread.
     * </p>
     */
    SPINYIELD ("spin+yield");

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new thread type instance with the given
     * human-readable name.
     * @param name the human-readable name for this thread type.
     */
    private ThreadType(final String name)
    {
        _textName = name;
    } // end of ThreadType(String)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Object Method Overrides.
    //

    @Override
    public String toString()
    {
        return (_textName);
    } // end of toString()

    //
    // end of Object Method Overrides.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get methods
    //

    /**
     * Returns the thread type text name.
     * @return text name.
     */
    public String textName()
    {
        return (_textName);
    } // end of textName()

    /**
     * Returns the thread type with a text name equaling
     * {@code s}, ignoring case. Will return {@code null} if
     * {@code s} does not match any thread type text name.
     * @param s compare the thread type text name to this given
     * value.
     * @return thread type for the given text name.
     */
    public static ThreadType find(final String s)
    {
        final ThreadType[] values = ThreadType.values();
        final int size = values.length;
        int i;
        ThreadType retval = null;

        for (i = 0; i < size && retval == null; ++i)
        {
            if ((values[i]._textName).equalsIgnoreCase(s))
            {
                retval = values[i];
            }
        }

        return (retval);
    } // end of find(String)

    //
    // end of Get methods.
    //-----------------------------------------------------------

//---------------------------------------------------------------
// Member data.
//

    /**
     * The name used in {@code toString} and eBus net property
     * values.
     */
    private final String _textName;
} // end of enum ThreadType
