//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2014, 2016. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.client;

import java.io.Serializable;
import java.net.InetSocketAddress;
import net.sf.eBus.messages.EMessageKey;

/**
 * This message reports the
 * {@link #remoteAddress remote Internet address and TCP port}
 * of a connection newly accepted by {@link EServer} but not yet
 * logged in.
 * <p>
 * All such messages have the subject
 * {@link AbstractEBusMessage#EBUS_SUBJECT /eBus}. Subscribe
 * to
 * {@link #MESSAGE_KEY ServerMessage.class:/eBus} to receive this
 * notification.
 * </p>
 * <p>
 * <strong>Note:</strong> this notification is published
 * locally only and is not sent to remote eBus applications.
 * </p>
 *
 * @see ConnectionMessage
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public final class ServerMessage
    extends AbstractEBusMessage
    implements Serializable
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * The server message key {@code ServerMessage.class}:/eBus.
     * Use this key when opening a system message
     * {@link net.sf.eBus.client.EFeed feed}.
     */
    public static final EMessageKey MESSAGE_KEY =
        new EMessageKey(ServerMessage.class, EBUS_SUBJECT);

    /**
     * Serialization version identifier.
     */
    private static final long serialVersionUID = 0x050200L;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new server accepted connection message for the
     * given remote eBus address. The subject is set to
     * {@link AbstractEBusMessage#EBUS_SUBJECT "/eBus"} and the
     * timestamp to the current time.
     * @param addr the newly accepted remote TCP address.
     * @param port connection accepted on this port.
     * @throws IllegalArgumentException
     * if {@code addr} is {@code null}.
     *
     * @deprecated use {@link Builder} to create server message.
     */
    @Deprecated
    public ServerMessage(final InetSocketAddress addr,
                         final int port)
        throws IllegalArgumentException
    {
        super (addr, port);
    } // end of ServerMessage(InetSocketAddress, int)

    /**
     * Creates a new server message for the given subject,
     * timestamp, and TCP address.
     * @param subject the notification subject. Will always be
     * {@link AbstractEBusMessage#EBUS_SUBJECT "/eBus"}.
     * @param timestamp the notification timestamp.
     * @param addr the newly accepted remote TCP address.
     * @param port connection accepted on this port.
     * @throws IllegalArgumentException
     * if {@code addr} is {@code null}.
     *
     * @deprecated use {@link Builder} to create server message.
     */
    @Deprecated
    public ServerMessage(final String subject,
                         final long timestamp,
                         final InetSocketAddress addr,
                         final int port)
        throws IllegalArgumentException
    {
        super (subject, timestamp, addr, port);
    } // end of ServerMessage(String,long,InetSocketAddress,int)

    /**
     * Creates a new server message based on the builder
     * settings.
     * @param builder server message builder.
     */
    private ServerMessage(final Builder builder)
    {
        super (builder);
    } // end of ServerMessage(Builder)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    /**
     * Returns a new server message builder instance.
     * @return message builder instance.
     */
    public static Builder builder()
    {
        return (new Builder());
    } // end of builder()

//---------------------------------------------------------------
// Inner classes.
//

    /**
     * Use this builder to create {@code ServerMessage}
     * instances.
     */
    public static final class Builder
        extends AbstractEBusMessage.Builder<ServerMessage, Builder>
    {
    //-----------------------------------------------------------
    // Member data.
    //

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        private Builder()
        {
            super (ServerMessage.class);
        } // end of Builder()

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Builder Method Overrides.
        //

        @Override
        protected ServerMessage buildImpl()
        {
            return (new ServerMessage(this));
        } // end of buildImpl()

        //
        // end of Builder Method Overrides.
        //-------------------------------------------------------
    } // end of class Builder
} // end of class ServerMessage
