//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2015, 2016. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.client;

import java.util.HashMap;
import java.util.Map;
import net.sf.eBus.client.sysmessages.KeyMessage;
import net.sf.eBus.client.sysmessages.SystemMessageType;
import net.sf.eBus.messages.EMessageHeader;
import net.sf.eBus.messages.EMessageKey;

/**
 * This class is responsible for mapping message classes to a
 * unique identifier (uniqueness scoped to within this JVM). This
 * mapping is shared with remote eBus applications upon
 * successful connection, where upon the remote eBus connection
 * maps the class identifier with a message reader. This message
 * reader contains both the message class de-serializer and
 * method handle to the message destination.
 * <p>
 * Using a class identifier in message transmission minimizes the
 * {@code ByteBuffer} de-serialization time and forwarding the
 * message to its ultimate destination. Without this feature,
 * eBus would have to insert the fully-qualified class name into
 * each message which takes more time and space to serialize and
 * de-serialize.
 * </p>
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

/* package */ final class MessageKeyStore
{
//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates the message class-to-identifier store for the
     * given eBus remote application connection.
     * @param connection remote connection.
     */
    /* package */ MessageKeyStore(final ERemoteApp connection)
    {
        _connection = connection;
        _ids = new HashMap<>();
        _nextId = 0;
    } // end of MessageKeyStore(ERemoteApp)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get Methods.
    //

    /**
     * Returns the identifier associated with the given message
     * class. If the message class is not in the message store,
     * then it is assigned the next available identifier and that
     * pairing is stored.
     * @param key return this message key.
     * @return unique message class identifier.
     */
    /* package */ synchronized int findOrCreate(final EMessageKey key)
    {
        int retval;

        if (_ids.containsKey(key))
        {
            retval = _ids.get(key);
        }
        else
        {
            retval = _nextId;
            _nextId++;
            _ids.put(key, retval);

            // Forward this new message class, identifier pair on
            // the remote eBus connection.
            _connection.send(
                new EMessageHeader(
                    (SystemMessageType.KEY_UPDATE).keyId(),
                    0,
                    0,
                    (KeyMessage.builder()).keyId(retval)
                                          .key(key)
                                          .build()));
        }

        return (retval);
    } // end of findOrCreate(EMessageKey)

    /**
     * Returns the class update messages based on the current,
     * local message class-to-identifier map entries. These
     * messages are sent to a remote eBus application.
     * @return current message class-to-identifier entries.
     */
    /* package */ synchronized KeyMessage[] messages()
    {
        final int size = _ids.size();
        int index = 0;
        final KeyMessage[] retval = new KeyMessage[size];

        _ids.entrySet()
            .forEach((entry) ->
            {
                retval[index] =
                    (KeyMessage.builder()).keyId(entry.getValue())
                                          .key(entry.getKey())
                                          .build();
            });

        return (retval);
    } // end of messages()

    //
    // end of Get Methods.
    //-----------------------------------------------------------

//---------------------------------------------------------------
// Member data.
//

    /**
     * This message class store belongs to this eBus remote
     * application connection. Forward newly defined message
     * class identifiers on this connection.
     */
    private final ERemoteApp _connection;

    /**
     * Maps the message class instance to its unique integer
     * identifier.
     */
    private final Map<EMessageKey, Integer> _ids;

    /**
     * Assign the next message class this identifier. Starts at
     * zero.
     */
    private int _nextId;
} // end of class MessageKeyStore
