//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2017. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.client;

import net.sf.eBus.messages.EReplyMessage;

/**
 * Provides a common view of single- and multi-key request feeds,
 * allowing such feeds to be referenced in a common way.
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public interface IERequestFeed
    extends IEFeed
{
//---------------------------------------------------------------
// Member methods.
//

    /**
     * Puts the feed status callback in place.
     * @param cb the feed status update callback. May be
     * {@code null}.
     */
    void statusCallback(FeedStatusCallback<ERequestFeed> cb);

    /**
     * Puts the reply message callback in place
     * <em>for all reply types</em>. If {@code cb} is
     * not {@code null}, replies will be passed to {@code cb}
     * rather than
     * {@link ERequestor#reply(int, EReplyMessage, ERequestFeed.ERequest)}.
     * A {@code null cb} results in replies posted to the
     * {@code ERequestor.reply(int, EReplyMessage, ERequestFeed.ERequest)}
     * override.
     * <p>
     * Note that this method call overrides all previous calls
     * to {@link #replyCallback(Class, ReplyCallback)}. If
     * the goal is to use a generic callback for all replies
     * except one specific message, then use this method to put
     * the generic callback in place first and then use
     * {@code replyCallback(EMessageKey, ReplyCallback)}.
     * </p>
     * @param cb reply message callback. May be {@code null}.
     * @throws IllegalStateException
     * if this feed is either closed or subscribed.
     *
     * @see #replyCallback(Class, ReplyCallback)
     */
    void replyCallback(final ReplyCallback cb);

    /**
     * Sets the callback for a specific reply message class. If
     * {@code cb} is not {@code null}, replies will be passed to
     * {@code cb} rather than
     * {@link ERequestor#reply(int, EReplyMessage, ERequestFeed.ERequest)}.
     * A {@code cb} results in replies posted to the
     * {@code ERequestor.reply(int, EReplyMessage, ERequestFeed.ERequest)}
     * override.
     * <p>
     * If the goal is to set a single callback method for all
     * reply message types, then use
     * {@link #replyCallback(ReplyCallback)}. Note that method
     * overrides all previous set reply callbacks.
     * </p>
     * @param mc the reply message class.
     * @param cb callback for the reply message.
     * @throws NullPointerException
     * if {@code mc} is {@code null}.
     * @throws IllegalArgumentException
     * if {@code mc} is not a reply for this request.
     * @throws IllegalStateException
     * if this feed is either closed or subscribed.
     */
    void replyCallback(final Class<? extends EReplyMessage> mc,
                       final ReplyCallback cb);

    /**
     * Activates this replier status subscription.
     */
    void subscribe();

    /**
     * De-activates this replier status subscription. Does
     * nothing if not currently subscribed.
     */
    void unsubscribe();
} // end of interface IERequestFeed

