//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2017. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.client;

/**
 * Provides a common view of single- and multi-key reply feeds,
 * allowing such feeds to be referenced in a common way.
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public interface IEReplyFeed
    extends IEFeed
{
//---------------------------------------------------------------
// Member methods.
//

    /**
     * Returns {@code true} if this reply feed is both open and
     * advertised; otherwise, returns {@code false}.
     * @return {@code true} if this reply feed is open and
     * advertised
     */
    boolean isAdvertised();

    /**
     * Puts the new request callback in place. If {@code cb} is
     * not {@code null}, requests will be passed to {@code cb}
     * rather than
     * {@link EReplier#request(EReplyFeed.ERequest)}. A
     * {@code null cb} means that requests are passed to the
     * {@link EReplier#request(EReplyFeed.ERequest)} override.
     * @param cb the request callback. May be {@code null}.
     * @throws IllegalStateException
     * if this feed is either closed or advertised.
     */
    void requestCallback(final RequestCallback cb);

    /**
     * Puts the cancel request callback in place. If {@code cb}
     * is not {@code null}, requests will be passed to {@code cb}
     * rather than
     * {@link EReplier#cancelRequest(EReplyFeed.ERequest)}. A
     * {@code null cb} means that cancellations are passed to the
     * {@link EReplier#cancelRequest(EReplyFeed.ERequest)}
     * override.
     * @param cb the cancel request callback. May be
     * {@code null}.
     * @throws IllegalStateException
     * if this feed is either closed or advertised.
     */
    void cancelRequestCallback(final CancelRequestCallback cb);

    /**
     * Advertises this replier fed to the associated request
     * subject. If this feed is currently advertised to the
     * subject, then does nothing.
     * @throws IllegalStateException
     * if this feed is closed or if the client did not override
     * {@link EReplier} methods nor put the required callbacks in
     * place.
     *
     * @see #unadvertise()
     * @see #updateFeedState(EFeedState)
     * @see EFeed#close()
     */
    void advertise();

    /**
     * Retracts this replier feed advertisement. Does nothing if
     * this feed is not currently advertised.
     * @throws IllegalStateException
     * if this reply feed is closed.
     *
     * @see #advertise()
     * @see #close()
     */
    public void unadvertise();

    /**
     * Updates the replier feed state to the given value. If
     * {@code update} equals the currently stored reply feed
     * state, nothing is done. Otherwise, the updated value is
     * stored. If this feed is advertised to the server and
     * the subscription feed is up, then this update is forwarded
     * to subscribed requestors.
     * @param update the new reply feed state.
     * @throws NullPointerException
     * if {@code update} is {@code null}.
     * @throws IllegalStateException
     * if this feed was closed and is inactive or is not
     * advertised.
     */
    public void updateFeedState(final EFeedState update);
} // end of interface IEReplyFeed

