//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2017. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.client;

import net.sf.eBus.messages.ENotificationMessage;

/**
 * Provides a common view for {@link EPublishFeed} and
 * {@link EMultiPublishFeed} feeds. Allows an application to
 * reference single- and multi-key publish feeds referenced as
 * an {@code IEPublishFeed}.
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public interface IEPublishFeed
    extends IEFeed
{
//---------------------------------------------------------------
// Member methods.
//

    /**
     * Returns {@code true} if this publish feed is both
     * open and advertised; otherwise, returns {@code false}.
     * @return {@code true} if this publish feed is open and
     * advertised.
     */
    boolean isAdvertised();

    /**
     * Returns the current publish state.
     * @return current publish state.
     */
    EFeedState publishState();

    /**
     * Returns {@code true} if the publisher is clear to publish
     * a notification for the given feed and {@code false} if not
     * clear.
     * @param subject check the feed status for this subject.
     * @return {@code true} if the specified publisher feed is up
     * and the publisher is free to publish notification
     * messages for the specified subject.
     */
    boolean isFeedUp(String subject);

    /**
     * Puts the publish status callback in place. If {@code cb}
     * is not {@code null}, publish status updates will be passed
     * to {@code cb} rather than
     * {@link EPublisher#publishStatus(EFeedState, IEPublishFeed)}.
     * The reverse is true if {@code cb} is {@code null}. That
     * is, a {@code null cb} means publish status updates are
     * posted to the
     * {@link EPublisher#publishStatus(EFeedState, IEPublishFeed)}
     * override.
     * @param cb publish status update callback. May be
     * {@code null}.
     */
    void statusCallback(FeedStatusCallback<IEPublishFeed> cb);

    /**
     * Advertises this publisher feed to the associated
     * notification subject. If this feed is currently advertised
     * to the subject, then does nothing.
     */
    void advertise();

    /**
     * Retracts this publisher feed from the associated
     * notification subject. Does nothing if this feed is not
     * currently advertised.
     */
    void unadvertise();

    /**
     * Updates the publish feed state to the given value. If
     * {@code update} equals the currently stored publish feed
     * state, nothing is done. Otherwise, the updated value is
     * stored. If this feed is advertised to the server and
     * the subscription feed is up, then this update is forwarded
     * to the subject.
     * @param update new publish feed state.
     */
    void updateFeedState(EFeedState update);

    /**
     * Posts this notification message to all interested
     * subscribers.
     * @param msg post this message to subscribers.
     */
    void publish(ENotificationMessage msg);
} // end of interface IEPublishFeed

