//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2017. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.client;

import net.sf.eBus.messages.EMessageKey;

/**
 * Common base interface for single- and multi-key feeds. Allows
 * feeds to be referenced in a common manner.
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public interface IEFeed
{
//---------------------------------------------------------------
// Member methods.
//

    /**
     * Returns the unique feed identifier. The uniqueness is
     * limited to within the client and for the feed lifespan
     * only. When a feed is closed, the feed identifier may be
     * reused.
     * @return the feed identifier.
     */
    int feedId();

    /**
     * Returns the feed scope.
     * @return feed scope.
     */
    EFeed.FeedScope scope();

    /**
     * Returns the eBus client referenced by this feed.
     * @return eBus client.
     */
    EClient eClient();

    /**
     * Returns {@code true} if this feed is still active and
     * {@code false} if inactive. Clients may only use active
     * feeds. Once a feed is closed, it is marked inactive and
     * may not be used by the client again.
     * @return {@code true} if this feed is active.
     */
    boolean isActive();

    /**
     * Returns {@code true} if this feed is "in place"
     * (advertised or subscribed) and {@code false} otherwise.
     * When this feed is closed, the feed is automatically
     * un-advertised/un-subscribed.
     * @return {@code true} if the feed is in place.
     */
    boolean inPlace();

    /**
     * Returns {@code true} if the feed is up and {@code false}
     * otherwise.
     * @return {@code true} if the feed is up.
     */
    boolean isFeedUp();

    /**
     * Returns the message key associated with this feed.
     * @return feed message key.
     */
    EMessageKey key();

    /**
     * Permanently closes the feed. After closing, this feed may
     * no longer be used.
     */
    void close();
} // end of interface IEFeed

