//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2016. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.client;

/**
 * An eBus client implementing either {@code ESubscriber},
 * {@code EPublisher}, or {@code ERequestor} is not required
 * to override the matching feed status method (
 * {@link ESubscriber#feedStatus(EFeedState, IESubscribeFeed)},
 * {@link EPublisher#publishStatus(EFeedState, IEPublishFeed)}, or
 * {@link ERequestor#feedStatus(EFeedState, ERequestFeed)}).
 * Instead, a lambda expression implementing this interface may
 * be used as the callback. This allows for tighter coupling
 * between the status callback and the code for handling the
 * callback.
 * <p>
 * Note: This must be done after opening a feed and before
 * advertising or subscribing.
 * </p>
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 *
 * @param <T> must be either {@link ERequestFeed},
 * {@link ESubscribeFeed}, or {@link EPublishFeed}.
 * @see EPublishFeed#statusCallback(FeedStatusCallback)
 * @see ESubscribeFeed#statusCallback(FeedStatusCallback)
 * @see ERequestFeed#statusCallback(FeedStatusCallback)
 */

@FunctionalInterface
public interface FeedStatusCallback<T extends IEFeed>
{
//---------------------------------------------------------------
// Member methods.
//

    /**
     * eBus calls this method to inform the eBus client that the
     * feed is either
     * {@link EFeedState#UP up} or {@link EFeedState#DOWN down}.
     * @param feedState the subscription's new feed state.
     * @param feed the status applies to this feed.
     */
    void call(EFeedState feedState, T feed);
} // end of interface FeedStatusCallback

