//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright (C) 2008, 2011, 2013, 2015, 2016. Charles W. Rapp.
// All Rights Reserved.
//

package net.sf.eBus.client;

import net.sf.eBus.messages.ENotificationMessage;

/**
 * Classes wanting to receive eBus notification messages need to
 * implement this interface. The application subscriber first
 * {@link ESubscribeFeed#open(ESubscriber, net.sf.eBus.messages.EMessageKey, EFeed.FeedScope, ECondition) opens}
 * the feed and {@link ESubscribeFeed#subscribe() subscribes} to
 * the notification. eBus calls
 * {@link #feedStatus(EFeedState, IESubscribeFeed)} to inform the
 * subscriber about whether the notification feed is
 * {@link EFeedState#UP up} or {@link EFeedState#DOWN down}. If
 * down, then the subscriber will not receive calls to
 * {@link #notify(ENotificationMessage, IESubscribeFeed)} until
 * the feed comes back up. If up, then subscriber may receive
 * notifications depending on whether the publisher(s) send any
 * messages. When a subscriber shuts down, it should
 * {@link ESubscribeFeed#unsubscribe() unsubscribe} or
 * {@link EFeed#close() close} the feed. If the subscriber does
 * not do this, eBus will automatically retract the subscription
 * when it detects the subscriber is finalized.
 * <p>
 * As of eBus v. 4.2.0, implementing the interface methods is no
 * longer necessary. Instead, Java lambda expressions may be used
 * to handle subscriber callbacks. This is done by calling
 * {@link ESubscribeFeed#statusCallback(FeedStatusCallback)}
 * and/or
 * {@link ESubscribeFeed#notifyCallback(NotifyCallback)} passing
 * in a lambda expression to specify the callback target.
 * Still, the application must either override {@code EReplier}
 * interface methods or put callbacks in place. Failure
 * to do either results in {@link ESubscribeFeed#subscribe()}
 * failing. A class wishing to subscribe to notification
 * messages must still implement {@code ESubscriber} even though
 * it is no longer necessary to override the interface methods.
 * </p>
 * <p>
 * It is possible to mix and match {@code ESubscriber} overrides
 * and callbacks. That is, a notify callback can be used in
 * conjunction with a
 * {@link ESubscriber#feedStatus(EFeedState, IESubscribeFeed)}
 * override.
 * </p>
 *
 * @see ENotifySubject
 * @see EPublisher
 * @see ESubscribeFeed
 * @see ENotificationMessage
 * @see IESubscribeFeed
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public interface ESubscriber
    extends EObject
{
    /**
     * eBus calls this method to inform the subscriber that the
     * subscribe notification feed is either
     * {@link EFeedState#UP up} or {@link EFeedState#DOWN down}.
     * If down, then
     * {@link #notify(ENotificationMessage, IESubscribeFeed)} will
     * not be called until the feed comes back up. If up, then
     * the subscriber will receive all notifications published on
     * {@code feed} until the subscription is retracted or the
     * feed goes down.
     * @param feedState the subscription's new feed state.
     * @param feed the status applies to this feed.
     * @throws UnsupportedOperationException
     * if implementing class does not override this method nor
     * uses a callback.
     *
     * @see #notify(ENotificationMessage, IESubscribeFeed)
     */
    default void feedStatus(EFeedState feedState,
                            IESubscribeFeed feed)
    {
        throw (
            new UnsupportedOperationException(
                "feedStatus not implemented"));
    } // end of feedStatus(EFeedState, ESubscribeFeed)

    /**
     * An incoming notification message from {@code feed}.
     * @param msg eBus notification message.
     * @param feed the associated subscription feed.
     * @throws UnsupportedOperationException
     * if implementing class does not override this method nor
     * uses a callback.
     *
     * @see #feedStatus(EFeedState, IESubscribeFeed)
     */
    default void notify(ENotificationMessage msg,
                        IESubscribeFeed feed)
    {
        throw (
            new UnsupportedOperationException(
                "notify not implemented"));
    } // end of notify(ENotificationMessage, ESubscribeFeed)
} // end of interface ESubscriber
