//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2018. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.client;

import java.util.LinkedList;
import java.util.List;
import net.sf.eBus.messages.EMessageKey;


/**
 * Base class for all feeds which connect to a single
 * {@link ESubject eBus subject}.
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

/* package */ abstract class ESingleFeed
    extends EFeed
{
//---------------------------------------------------------------
// Enums.
//

    /**
     * Enumerates the supported feed types. There are four basic
     * feed types:
     * <ol>
     *   <li>
     *     publishing notification messages,
     *   </li>
     *   <li>
     *     receiving notification messages,
     *   </li>
     *   <li>
     *     sending a request message, and
     *   </li>
     *   <li>
     *     replying to a request message.
     *   </li>
     * </ol>
     */
    protected enum FeedType
    {
        /**
         * Feed for publishing notification messages.
         */
        PUBLISH_FEED,

        /**
         * Feed for receiving notification messages.
         */
        SUBSCRIBE_FEED,

        /**
         * Feed for placing requests.
         */
        REQUEST_FEED,

        /**
         * Feed for replying to requests.
         */
        REPLY_FEED
    } // end of enum FeedType

//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Statics.
    //

    /**
     * Track the advertised feeds in order to generate
     * {@link net.sf.eBus.client.sysmessages.AdMessage advertise messages}.
     */
    protected static final List<EFeed> mAdvertisers =
        new LinkedList<>();

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * Specifies whether this is a publish, subscribe, request,
     * or reply feed.
     */
    protected final FeedType mFeedType;

    /**
     * The feed interfaces with this eBus subject. The subject
     * type is based on {@link #mFeedType feed type}.
     */
    protected final ESubject mSubject;

    /**
     * Tracks the number of contra-feeds matched to this feed. If
     * this is a publisher feed, then counts subscriber feeds. If
     * a subscriber feed, then counts publisher feeds.
     */
    protected int mActivationCount;

    /**
     * Index in the {@link EFeedList}. Note that a feed is stored
     * in only one feed list.
     */
    private int mFeedIndex;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new single eBus feed instance for the given
     * parameters.
     * @param client post eBus tasks to this client.
     * @param feedScope this feed supports either local, local
     * &amp; remote, or just remote feeds.
     * @param feedType the actual feed type.
     * @param subject interact with this eBus subject.
     */
    protected ESingleFeed(final EClient client,
                          final FeedScope feedScope,
                          final FeedType feedType,
                          final ESubject subject)
    {
        super (client, feedScope);

        mFeedType = feedType;
        mSubject = subject;
        mFeedIndex = -1;
        mActivationCount = 0;
    } // end of ESingleFeed()

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Abstract Method Declarations.
    //

    /**
     * Updates the feed activation count <em>if</em> the feed
     * scope supports the contra-feed location. If the activation
     * count transitions between activate and inactive, then
     * updates the feed.
     * @param loc contra-feed location.
     * @param fs contra-feed state.
     * @return 1 if activated, -1 if deactivated, and zero if
     * not affected.
     */
    /* package */ abstract int
        updateActivation(final EClient.ClientLocation loc,
                         final EFeedState fs);

    //
    // end of Abstract Method Declarations.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Object Method Overrides.
    //

    /**
     * Returns a containing the feed message key and data member
     * values.
     * @return textual representation of this feed.
     */
    @Override
    public String toString()
    {
        return (String.format("%s %s %s",
                              mSubject.key(),
                              mFeedType,
                              super.toString()));
    } // end of toString()

    //
    // end of Object Method Overrides.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get Methods.
    //

    /**
     * Returns the feed message key. The message key type matches
     * the feed type. If this is a {@link EPublishFeed}, then a
     * {@link net.sf.eBus.messages.ENotificationMessage notification}
     * message key is returned.
     * @return message key.
     */
    @Override
    public final EMessageKey key()
    {
        return (mSubject.key());
    } // end of key()

    /**
     * Returns the feed
     * {@link EMessageKey#subject() message key subect}.
     * @return message key subject.
     */
    public final String messageSubject()
    {
        return ((mSubject.key()).subject());
    } // end of messageSubject()

    /**
     * Returns the index into the {@link EFeedList}.
     * @return feed list index.
     */
    /* package */ final int feedIndex()
    {
        return (mFeedIndex);
    } // end of feedIndex()

    /**
     * Returns the feed activation count.
     * @return value &ge; zero.
     */
    public final int activationCount()
    {
        return (mActivationCount);
    } // end of activationCount()

    //
    // end of Get Methods.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Set Methods.
    //

    /**
     * Sets the {@link EFeedList feed list} index to the given
     * value.
     * @param index the updated feed list index.
     */
    /* package */ final void feedIndex(final int index)
    {
        mFeedIndex = index;
        return;
    } // end of feedIndex(int)

    //
    // end of Set Methods.
    //-----------------------------------------------------------
} // end of class ESingleFeed
