//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright (C) 2008 - 2011, 2013, 2015, 2016. Charles W. Rapp.
// All Rights Reserved.
//

package net.sf.eBus.client;

import net.sf.eBus.messages.EReplyMessage;

/**
 * Classes wanting to send eBus requests must implement this
 * interface. A requestor must first successfully
 * {@link ERequestFeed#open(ERequestor, net.sf.eBus.messages.EMessageKey, EFeed.FeedScope) open}
 * a request feed and then
 * {@link ERequestFeed#request(net.sf.eBus.messages.ERequestMessage) send}
 * a request message. eBus forwards the request to all repliers
 * matching the given request. If there are no matching repliers
 * for the request, then {@code ERequestFeed.request} throws an
 * {@code IllegalStateexception} with the message "no repliers
 * for request". Otherwise, returns a
 * {@link ERequestFeed.ERequest} feed instance. At this point,
 * the requestor may expect to receive reply messages on the
 * {@link #reply(int, EReplyMessage, ERequestFeed.ERequest)}
 * callback (or its replacement).
 * <p>
 * A request may be forwarded to more than one replier. Each
 * replier may send more than one reply. A requestor knows when a
 * final reply is received when the {@code remaining} parameter
 * is zero.
 * </p>
 * <p>
 * A requestor may cancel a request any time before the final
 * reply is received by calling
 * {@link EFeed#close() ERequestFeed.ERequest.close()}. The
 * requestor may receive replies after this point if the reply
 * messages were scheduled for delivery prior to canceling the
 * request.
 * </p>
 * <p>
 * As of eBus v. 4.2.0, implementing the interface methods is no
 * longer necessary. Instead, Java lambda expressions may be used
 * to handle requestor callbacks. This is done by calling
 * {@link ERequestFeed#statusCallback(FeedStatusCallback)} and/or
 * {@link ERequestFeed#replyCallback(ReplyCallback)}
 * and using a lambda expression to specify the callback target.
 * Still, the application must either override {@code ERequestor}
 * interface methods or put callbacks in place. Failure
 * to do either results in {@link ERequestFeed#subscribe()}
 * failing. A class wishing to post requests must still implement
 * {@code ERequestor} even though it is no longer necessary to
 * override the interface methods.
 * </p>
 * <p>
 * It is possible to mix and match {@code EReplier} overrides and
 * callbacks. That is, a reply callback can be used in
 * conjunction with a
 * {@link ERequestor#feedStatus(EFeedState, ERequestFeed)}
 * override.
 * </p>
 *
 * @see EReplier
 * @see ERequestFeed.ERequest
 * @see ERequestFeed
 * @see net.sf.eBus.messages.ERequestMessage
 * @see EReplyMessage
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public interface ERequestor
    extends EObject
{
    /**
     * eBus calls this method to inform the requestor that the
     * request feed is either has at least one replier
     * ({@link EFeedState#UP up}) or no repliers
     * ({@link EFeedState#DOWN down}).
     * If down, then any call to
     * {@link ERequestFeed#request(ERequestMessage)} will result
     * in an {@code IllegalStateException}. If up, then a
     * request will attempt to post to the advertised repliers.
     * <p>
     * Note that an {@code IllegalStateException} may still be
     * thrown when the feed state is up because none of the
     * repliers accepted the request.
     * </p>
     * @param feedState the subscription's new feed state.
     * @param feed the status applies to this feed.
     * @throws UnsupportedOperationException
     * if implementing class does not override this method nor
     * uses a callback.
     *
     * @see #reply(int, EReplyMessage, ERequestFeed.ERequest)
     */
    default void feedStatus(EFeedState feedState,
                            ERequestFeed feed)
    {
        throw (
            new UnsupportedOperationException(
                "feedStatus not implemented"));
    } // end of feedStatus(EFeedState, IERequestFeed)

    /**
     * eBus calls this method to deliver a reply message to the
     * corresponding {@code request}. The requestor can determine
     * that this is the final reply to the request when
     * {@code remaining} is zero.
     * @param remaining the number of replies yet to be received.
     * Zero means all replies received and no more will be
     * forthcoming.
     * @param reply the latest reply.
     * @param request the reply is for this request.
     * @throws UnsupportedOperationException
     * if implementing class does not override this method nor
     * uses a callback.
     *
     * @see #feedStatus(EFeedState, ERequestFeed)
     */
    default void reply(int remaining,
                       EReplyMessage reply,
                       ERequestFeed.ERequest request)
    {
        throw (
            new UnsupportedOperationException(
                "reply not implemented"));
    } // end of reply(int, EReplyMessage, ERequestFeed.ERequest)
} // end of interface ERequestor
