//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright (C) 2008 - 2011, 2013, 2015, 2016. Charles W. Rapp.
// All Rights Reserved.
//

package net.sf.eBus.client;

/**
 * Classes able to reply to eBus requests must implement this
 * interface. A replier must:
 * <ol>
 *   <li>
 *     {@link EReplyFeed#open(EReplier, net.sf.eBus.messages.EMessageKey, EFeed.FeedScope, ECondition) open}
 *     the reply feed,
 *   </li>
 *   <li>
 *     {@link EReplyFeed#advertise() advertise} this reply ability, and
 *   </li>
 *   <li>
 *     {@link EReplyFeed#updateFeedState(EFeedState) declare}
 *     with an up feed state that it (the replier) is ready to
 *     handle requests.
 *   </li>
 * </ol>
 * The message key used in opening the feed must be a
 * <em>request</em> message class.
 * <p>
 * A replier can retract an advertisement by calling
 * {@link EReplyFeed#unadvertise()}. The reply feed is discarded
 * by calling {@link EReplyFeed#close()}. If a replier is
 * temporarily unable to reply to a request, then
 * {@code updateFeedState(EFeedState.DOWN)} should be used rather
 * than un-advertising. When the replier is again able to
 * respond, {@code updateFeedState(EFeedState.UP)} is called.
 * </p>
 * <p>
 * Requests matching the advertised reply feed are forwarded as
 * {@link EReplyFeed.ERequest} instances via the
 * {@link EReplier#request(EReplyFeed.ERequest)}
 * callback. Replies are sent via
 * {@link EReplyFeed.ERequest#reply(net.sf.eBus.messages.EReplyMessage)}.
 * </p>
 * <p>
 * <strong>Note:</strong> the replier is limited to
 * {@link net.sf.eBus.messages.EReplyMessage reply messages}
 * specified in the
 * {@link net.sf.eBus.messages.ERequestMessage request message's}
 * {@link net.sf.eBus.messages.EReplyInfo} annotation or in the
 * request message super class ({@code EReplyInfo} is inherited).
 * If the reply message is not a member of
 * {@link net.sf.eBus.messages.EReplyInfo#replyMessageClasses()},
 * then it will be rejected.
 * </p>
 * <p>
 * If a requestor cancels a request, the replier is informed by
 * the {@link EReplier#cancelRequest(EReplyFeed.ERequest)}
 * callback. The request is unilaterally ended by the
 * {@link ERequestor requestor} and the replier may no longer
 * send replies to the {@link EReplyFeed.ERequest request}.
 * </p>
 * <p>
 * As of eBus v. 4.2.0, implementing the interface methods is no
 * longer necessary. Instead, Java lambda expressions may be used
 * to handle replier callbacks. This is done by calling
 * {@link EReplyFeed#requestCallback(RequestCallback)} and/or
 * {@link EReplyFeed#cancelRequestCallback(CancelRequestCallback)}
 * and passing a lambda expression to specify the callback
 * target. Still, the application must either override
 * {@code EReplier} interface methods or put callbacks in place.
 * Failure to do either results in {@link EReplyFeed#advertise()}
 * failing. A class wishing to receive requests must still
 * implement {@code EReplier} even though it is no longer
 * necessary to override the interface methods.
 * </p>
 * <p>
 * It is possible to mix and match {@code EReplier} overrides and
 * callbacks. That is, a request callback can be used in
 * conjunction with a
 * {@link EReplier#cancelRequest(EReplyFeed.ERequest)} override.
 * </p>
 * <p>
 * eBus maintains a weak reference to application objects. If a
 * replier is garbage collected without un-advertising and
 * closing its reply feed, eBus automatically closes the feed and
 * posts failure replies to all active requests.
 * </p>
 *
 * @see ERequestor
 * @see EReplyFeed.ERequest
 * @see EReplyFeed
 * @see net.sf.eBus.messages.ERequestMessage
 * @see net.sf.eBus.messages.EReplyMessage
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public interface EReplier
    extends EObject
{
    /**
     * An incoming request. The replier may send a
     * {@link EReplyFeed.ERequest#reply(net.sf.eBus.messages.EReplyMessage) reply}
     * either from within this method call or asynchronously
     * after returning from this method. If the reply is sent
     * asynchronously, then the replier must store
     * {@code request} for later use. Replies are sent using
     * {@code ERequest} and not {@code EReplyFeed}.
     * <p>
     * The
     * {@link net.sf.eBus.messages.ERequestMessage request message}
     * is stored in {@code request} and can be retrieved by
     * calling {@link EReplyFeed.ERequest#request()}.
     * </p>
     * <p>
     * The {@link EReplyFeed.ERequest request} matches the
     * replier's open and advertised {@link EReplyFeed feed}. The
     * associated {@code EReplyFeed} may be retrieved by calling
     * {@link EReplyFeed.ERequest#replier()}.
     * </p>
     * @param request post replies via this request.
     * @throws UnsupportedOperationException
     * if implementing class does not override this method nor
     * uses a callback.
     *
     * @see #cancelRequest(EReplyFeed.ERequest)
     */
    default void request(EReplyFeed.ERequest request)
    {
        throw (
            new UnsupportedOperationException(
                "request not implemented"));
    } // end of request(EReplyFeed.ERequest)

    /**
     * The specified request is canceled by the
     * {@link ERequestor requestor}. The request is shut down and
     * no further replies will be accepted. However, in-flight
     * replies posted prior to cancellation <em>may</em> still be
     * delivered.
     * <p>
     * The {@link EReplyFeed.ERequest request} matches the
     * replier's open and advertised {@link EReplyFeed feed}. The
     * associated {@code EReplyFeed} may be retrieved by calling
     * {@link EReplyFeed.ERequest#replier()}.
     * </p>
     * @param request cancel this request.
     * @throws UnsupportedOperationException
     * if implementing class does not override this method nor
     * uses a callback.
     *
     * @see #request(EReplyFeed.ERequest)
     */
    default void cancelRequest(EReplyFeed.ERequest request)
    {
        throw (
            new UnsupportedOperationException(
                "cancelRequest not implemented"));
    } // end of cancelRequest(EReplyFeed.ERequest)
} // end of interface EReplier
